/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
#include <stdint.h>
#include <string.h>
#include "board_cfg.h"
#include "boot_low_level_init.h"
#include "boot_clock.h"
#include "boot_delay.h"
#include "boot_sysctl_subcrg_clock_v2_2.h"
#include "device.h"
#include "driverlib.h"
#include "bootrom_select_bootmode.h"
#include "bootloader_sci.h"
#include "bootloader_can.h"
#include "bootrom.h"

uint32_t bootMode;
uint32_t bootAdress;

void fdp_jump_to_address(uintptr_t addr)
{
	asm volatile ("jr %0" :: "r"(addr));
}

int main(void)
{
	boot_low_level_init(DEVICE_SYSCLK_FREQ);

	/* Enable peripheral. */
	SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_CAN);
	SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_SCI);
	SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TIMER );

	SysCtl_resetCan();
	SysCtl_resetSci();
	SysCtl_resetCpuTimer();

	boot_sysctl_clock_init(DEVICE_SYSCLK_FREQ*2, DEVICE_OSCSRC_FREQ, DEVICE_SYSCLK_DIV, DEVICE_AHBCLK_DIV, DEVICE_APBCLK_DIV, __HSE_ENABLE, __HSE_CLOCK_TYPE);

	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CAN);
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SCI);
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TIMER);

	boot_delay_init(DEVICE_SYSCLK_FREQ);

	DINT;

	bootMode = BROM_getStandaloneBootMode(HWREAD_Z2_OTP_BOOTPIN_CONFIG_KEY, HWREAD_Z1_OTP_BOOTPIN_CONFIG_KEY);

    //
    // Run selected boot mode
    //
    switch(bootMode)
    {
        case SCI_BOOT:
        case SCI_BOOT_ALT1:
        case SCI_BOOT_ALT2:
        case SCI_BOOT_ALT3:
        case SCI_BOOT_ALT4:
        case SCI_BOOT_ALT5:
        case SCI_BOOT_ALT6:
        	bootAdress = SCI_Boot(bootMode);
            break;

        case CAN_BOOT:
        case CAN_BOOT_ALT1:
        case CAN_BOOT_ALT2:
        case CAN_BOOT_ALT3:
        case CAN_BOOT_ALT4:
        case CAN_BOOT_SENDTEST:
        case CAN_BOOT_ALT1_SENDTEST:
        case CAN_BOOT_ALT2_SENDTEST:
        	bootAdress = DCAN_Boot(bootMode);
            break;

        default:
        	break;
    }
	MFlushDCache();
	DELAY_US(2000);
	fdp_jump_to_address(bootAdress);
	while(1);

	return 0;
}
