/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "m_can_dev.h"

#include <stdint.h>
#include <string.h>

#include "fdp_err.h"
#include "fdp.h"

#include "device.h"
#include "mcan.h"
#include "bootrom_PIN.h"
#include "boot_gpio.h"

uint32_t M_CAN_BASE;
uint32_t M_CAN_IRQ_NUM;
uint32_t M_CAN_IRQ_NUM1;

#define M_CAN_MSG_MAX_LEN				CAN_DATA_LENGTH_8

#define M_CAN_MSG_MAX_LENGTH			MCAN_ELEM_SIZE_8BYTES
#define M_CAN_MSG_BUFFER_SIZE			(MCAN_getMsgObjSize(M_CAN_MSG_MAX_LENGTH) + 8U)

#define M_CAN_RAM_BASE_ADDR				0U
#define M_CAN_STANDARD_FILTER_BASE_ADDR	M_CAN_RAM_BASE_ADDR
#define M_CAN_STANDARD_FILTER_NUM		(4U)
#define M_CAN_STANDARD_FILTER_SIZE		(4U)

/* Note: The size of all memory regions cannot exceed 4096 bytes. */
#define M_CAN_EXTENDED_FILTER_BASE_ADDR	(M_CAN_STANDARD_FILTER_BASE_ADDR + \
										(M_CAN_STANDARD_FILTER_NUM * \
										 M_CAN_STANDARD_FILTER_SIZE))
#define M_CAN_EXTENDED_FILTER_NUM		(1U)
#define M_CAN_EXTENDED_FILTER_SIZE		(8U)

#define M_CAN_RXFIFO0_BASE_ADDR			(M_CAN_EXTENDED_FILTER_BASE_ADDR + \
										(M_CAN_EXTENDED_FILTER_NUM * \
										 M_CAN_EXTENDED_FILTER_SIZE))
#define M_CAN_RXFIFO0_NUM				(16U)
#define M_CAN_RXFIFO0_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_RXFIFO1_BASE_ADDR			(M_CAN_RXFIFO0_BASE_ADDR + \
										(M_CAN_RXFIFO0_NUM * \
										 M_CAN_RXFIFO0_SIZE))
#define M_CAN_RXFIFO1_NUM				(0U)
#define M_CAN_RXFIFO1_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_RXBUFFER_BASE_ADDR		(M_CAN_RXFIFO1_BASE_ADDR + \
										(M_CAN_RXFIFO1_NUM * \
										 M_CAN_RXFIFO1_SIZE))
#define M_CAN_RXBUFFER_NUM				(0U)
#define M_CAN_RXBUFFER_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_TXEVENTFIFO_BASE_ADDR		(M_CAN_RXBUFFER_BASE_ADDR + \
										(M_CAN_RXBUFFER_NUM * \
										 M_CAN_RXBUFFER_SIZE))
#define M_CAN_TXEVENTFIFO_NUM			(0U)
#define M_CAN_TXEVENTFIFO_SIZE			(M_CAN_TX_EVENT_FIFO_SIZE)

#define M_CAN_TXBUFFER_BASE_ADDR		(M_CAN_TXEVENTFIFO_BASE_ADDR + \
										(M_CAN_TXEVENTFIFO_NUM * \
										 M_CAN_TXEVENTFIFO_SIZE))
#define M_CAN_TXBUFFER_NUM				(0U)
#define M_CAN_TXBUFFER_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_TXFIFOQUEUE_BASE_ADDR		(M_CAN_TXBUFFER_BASE_ADDR + \
										(M_CAN_TXBUFFER_NUM * \
										 M_CAN_TXBUFFER_SIZE))
#define M_CAN_TXFIFOQUEUE_NUM			(16U)
#define M_CAN_TXFIFOQUEUE_SIZE			(M_CAN_MSG_BUFFER_SIZE)

/* extern bootromPIN_t *bootromPIN; */
extern bootromPIN_t *bootromPIN;

void M_CAN_interrupt_irq_handler(void)
{
	uint32_t status;
	MCAN_RxFIFOStatus_t fifoStatus;

	status = MCAN_getIntrStatus(M_CAN_BASE);

	if (status & MCAN_INT_SRC_RxFIFO0_NEW_MSG) {
		fdp_receive_data_to_buffer();
	}

	fifoStatus.num = MCAN_RX_FIFO_NUM_0;
	MCAN_getRxFIFOStatus(M_CAN_BASE, &fifoStatus);

	if (fifoStatus.fillLvl == 0)
		MCAN_clearIntrStatus(M_CAN_BASE, MCAN_INT_SRC_RxFIFO0_NEW_MSG);
}

static void M_CAN_config(void)
{
	MCAN_InitParams_t init_param;
	MCAN_ConfigParams_t configParams;
	MCAN_MsgRAMConfigParams_t RAMConfig;
	MCAN_BitTimingParams_t configBitrate;
	MCAN_StdMsgIDFilterElement_t std_filter;
	MCAN_ExtMsgIDFilterElement_t ext_filter;

	/* CAN mode configuration. */
	init_param.fdMode = false;
	init_param.fdFormat = MCAN_FD_ISO_11898_1;
	init_param.brsEnable = false;
	init_param.txpEnable = false;
	init_param.efbi = false;
	init_param.pxhddisable = false;
	/* Enable the auto retransmission. */
	init_param.darEnable = true;
	init_param.wkupReqEnable = false;
	init_param.autoWkupEnable = false;
	init_param.emulationEnable = false;
	init_param.wdcPreload = false;
	init_param.wmMarker = MCAN_WMM_8BIT_MODE;
	init_param.tdcEnable = false;
	init_param.tdcConfig.tdcf = 0U;
	init_param.tdcConfig.tdco = 0U;

	/* Set CAN extended feature. */
	configParams.asmEnable = false;
	configParams.monEnable = false;
	configParams.timeoutCntEnable = false;
	configParams.timeoutPreload = 1;
	configParams.timeoutSelect = MCAN_TIMEOUT_SELECT_CONT;

	configParams.tsClock = MCAN_INTERNAL_TIMESTAMP;
	configParams.tsSelect = MCAN_TSCNTVAL_ALWAYS0;
	configParams.tsPrescalar = 1;

	/* Set global configuration. */
	configParams.filterConfig.rrfs = true;
	configParams.filterConfig.rrfe = true;
	configParams.filterConfig.anfs = MCAN_NON_MATCH_REJECT_ACCEPT;
	configParams.filterConfig.anfe = MCAN_NON_MATCH_REJECT_ACCEPT;

	/* Set standard ID filter. */
	RAMConfig.lss = M_CAN_STANDARD_FILTER_NUM;
	RAMConfig.flssa = M_CAN_STANDARD_FILTER_BASE_ADDR;

	/* Set extended ID filter. */
	RAMConfig.lse = M_CAN_EXTENDED_FILTER_NUM;
	RAMConfig.flesa = M_CAN_EXTENDED_FILTER_BASE_ADDR;

	/* Set Rx fifo 0. */
	RAMConfig.rxFIFO0OpMode = MCAN_RXFIFO_OP_MODE_BLOCKING;
	RAMConfig.rxFIFO0startAddr = M_CAN_RXFIFO0_BASE_ADDR;
	RAMConfig.rxFIFO0size = M_CAN_RXFIFO0_NUM;
	RAMConfig.rxFIFO0waterMark = 0;
	RAMConfig.rxFIFO0ElemSize  = M_CAN_MSG_MAX_LENGTH;

	/* Set Rx fifo 1. */
	RAMConfig.rxFIFO1OpMode = MCAN_RXFIFO_OP_MODE_BLOCKING;
	RAMConfig.rxFIFO1startAddr = M_CAN_RXFIFO1_BASE_ADDR;
	RAMConfig.rxFIFO1size = M_CAN_RXFIFO1_NUM;
	RAMConfig.rxFIFO1waterMark = 0;
	RAMConfig.rxFIFO1ElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Rx buffer. */
	RAMConfig.rxBufStartAddr = M_CAN_RXBUFFER_BASE_ADDR;
	RAMConfig.rxBufElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Tx buffer */
	RAMConfig.txBufMode = MCAN_TXBUF_OP_IN_FIFO_MODE;
	RAMConfig.txStartAddr = M_CAN_TXBUFFER_BASE_ADDR;
	RAMConfig.txFIFOSize = M_CAN_TXFIFOQUEUE_NUM;
	RAMConfig.txBufNum = M_CAN_TXBUFFER_NUM;
	RAMConfig.txBufElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Tx event fifo. */
	RAMConfig.txEventFIFOStartAddr = M_CAN_TXEVENTFIFO_BASE_ADDR;
	RAMConfig.txEventFIFOSize = M_CAN_TXEVENTFIFO_NUM;
	RAMConfig.txEventFIFOWaterMark = 1;

	/* Input clock source 40Mhz, normal phase bitrate 500kbps. */
	configBitrate.nomRatePrescalar = 1;
	configBitrate.nomTimeSeg1 = 30;
	configBitrate.nomTimeSeg2 = 7;
	configBitrate.nomSynchJumpWidth = 1;

	/* Reset the MCAN module. */
	while (MCAN_isInReset(M_CAN_BASE));

	/* Check if the MCAN RAM is ready. */
	while (!MCAN_isMemInitDone(M_CAN_BASE));

	/* Set the MCAN mode to init mode. */
	if (!MCAN_setOpMode(M_CAN_BASE, MCAN_OPERATION_SW_INIT_MODE))
		while (1);

	/* Initialize the MCAN. */
	if (!MCAN_init(M_CAN_BASE, &init_param))
		while (1);

	if (!MCAN_config(M_CAN_BASE, &configParams))
		while (1);

	if (!MCAN_setBitTime(M_CAN_BASE, &configBitrate))
		while (1);

	if (!MCAN_msgRAMConfig(M_CAN_BASE, &RAMConfig))
		while (1);

	/* Add a new standard ID filter(Use Rx buffer 0). */
	std_filter.sfid1 = bootromPIN->CAN_RX_ID;
	std_filter.sfid2 = 0x7ff;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_FO0_OF_MATCH_ID;
	std_filter.sft = MCAN_STDIDF_CLASSIC_ID_FILTER;
	if (!MCAN_addStdMsgIDFilter(M_CAN_BASE, 0, &std_filter))
		while (1);

	/* The received extension frame will be subjected to AND operation with Ext_mask. */
	if (!MCAN_setExtIDAndMask(M_CAN_BASE, 0x1fffffff))
		while (1);

	/* Enable/Disable Loopback mode. */
	if (!MCAN_lpbkModeEnable(M_CAN_BASE, MCAN_LPBK_MODE_INTERNAL, false))
		while (1);

	/* Enable MCAN. */
	if (!MCAN_setOpMode(M_CAN_BASE, MCAN_OPERATION_NORMAL_MODE))
		while (1);

	 /* Configuration the external timestamp clock source. */
	 if (!MCAN_extTSCounterConfig(M_CAN_BASE, 0xffff))
	 	while (1);

	 /* Enable/Disable external timestamp clock source. */
	 if (!MCAN_extTSCounterEnable(M_CAN_BASE, true))
	 	while (1);

	 /* Enable/Disable external timestamp overflow interrupt. */
	 if (!MCAN_extTSEnableIntr(M_CAN_BASE, false))
	 	while (1);

	/* Select MCAN interrupt route to interrupt line 0 or 1. */
	if (!MCAN_selectIntrLine(M_CAN_BASE, MCAN_INT_SRC_RxFIFO0_NEW_MSG,
									   MCAN_INTERRUPT_LINE_0))
		while (1);

	/* Enable MCAN interrupt. */
	if (!MCAN_enableIntr(M_CAN_BASE, MCAN_INT_SRC_RxFIFO0_NEW_MSG, true))
		while (1);

	if (!MCAN_enableIntrLine(M_CAN_BASE, MCAN_INTERRUPT_LINE_0, true))
		while (1);

	if (!MCAN_enableIntrLine(M_CAN_BASE, MCAN_INTERRUPT_LINE_1, true))
		while (1);
}

void M_CAN_set_pinmux(void)
{
	boot_gpio_set_pin_config(bootromPIN->CAN_TX_PINMUX_Value);
	boot_gpio_set_pin_config(bootromPIN->CAN_RX_PINMUX_Value);

	boot_gpio_set_analog_mode(bootromPIN->CAN_TX_PIN_NUM_Value, GPIO_ANALOG_DISABLED);
	boot_gpio_set_analog_mode(bootromPIN->CAN_RX_PIN_NUM_Value, GPIO_ANALOG_DISABLED);

	boot_gpio_set_pad_config(bootromPIN->CAN_TX_PIN_NUM_Value, GPIO_PIN_TYPE_PULLUP);
	boot_gpio_set_pad_config(bootromPIN->CAN_RX_PIN_NUM_Value, GPIO_PIN_TYPE_PULLUP);
}

int M_CAN_Init(void)
{
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CAN);
	SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);
	SysCtl_setCanClkDiv(DEVICE_PLLCLK_FREQ / 40000000U);
	SysCtl_resetCana();
	SysCtl_resetCanb();

	M_CAN_set_pinmux();

	if(bootromPIN->CAN_number == 0){
		M_CAN_BASE    = CANA_BASE;
		M_CAN_IRQ_NUM = INT_CANA0;
		M_CAN_IRQ_NUM1 = INT_CANA1;
	} else if (bootromPIN->CAN_number == 1){
		M_CAN_BASE    = CANB_BASE;
		M_CAN_IRQ_NUM = INT_CANB0;
		M_CAN_IRQ_NUM1 = INT_CANB1;
	} else {
		return -FDP_ERRINPARAM;
	}

	M_CAN_config();

#if (FDP_IS_USE_INTERRUPT)
	Interrupt_register(M_CAN_IRQ_NUM, can_interrupt_irq_handler);
	Interrupt_enable(M_CAN_IRQ_NUM);

	Interrupt_register(M_CAN_IRQ_NUM1, can_interrupt_irq_handler);
	Interrupt_enable(M_CAN_IRQ_NUM1);
#endif

	return FDP_SUCCESS;
}

int M_CAN_Deinit(void)
{
	if (!MCAN_setOpMode(M_CAN_BASE, MCAN_OPERATION_SW_INIT_MODE))
		return -FDP_CHECKERR;

#if (FDP_IS_USE_INTERRUPT)
	Interrupt_unregister(CAN_CTRL_IRQ_NUM);
	Interrupt_disable(CAN_CTRL_IRQ_NUM);
#endif

	return FDP_SUCCESS;
}

int M_CAN_Receiver(uint8_t *buf, uint32_t *index, uint32_t buf_len)
{
	MCAN_RxMessage_t RxMsgBuf = {0};
	uint32_t len = 0;
	while(MCAN_receiveMsgFromFifo0(M_CAN_BASE, &RxMsgBuf)) {
		len = MCAN_getDataSize(RxMsgBuf.dlc);
		for (uint32_t i = 0; i < len; i++) {
			buf[(*index)++] = RxMsgBuf.data[i];
			if (*index >= buf_len)
				*index = 0;
		}
	}

	return FDP_SUCCESS;
}

int M_CAN_Transmit(uint8_t *buf, uint32_t len)
{
	MCAN_TxMessage_t TxMsgBuf = {0};
	uint32_t i = 0;
	int datalen, maxlen;

	TxMsgBuf.brs = 0;
	TxMsgBuf.dlc = MCAN_DATA_LENGTH_8;
	TxMsgBuf.efc = 0;
	TxMsgBuf.esi = 0;
	TxMsgBuf.fdf = 0;
	TxMsgBuf.id_std = bootromPIN->CAN_TX_ID;
	TxMsgBuf.mm = 0x0;
	TxMsgBuf.mm1 = 0x0;
	TxMsgBuf.rtr = 0;
	TxMsgBuf.tsce = 0;
	TxMsgBuf.xtd = 0;

	if (len == 0)
		return FDP_SUCCESS;

	maxlen = MCAN_getDataSize(M_CAN_MSG_MAX_LEN);

	while (len) {

		if (len >= maxlen) {
			TxMsgBuf.dlc = M_CAN_MSG_MAX_LEN;
			len -= maxlen;
		} else {
			TxMsgBuf.dlc = len;
			len = 0;
		}

		datalen = MCAN_getDataSize(TxMsgBuf.dlc);
		memcpy(TxMsgBuf.data, &buf[i], datalen);
		i += datalen;

		while(!MCAN_transmitMsgFifoQueue(M_CAN_BASE, &TxMsgBuf));

	}

	return FDP_SUCCESS;
}
