/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "boot_hw_type.h"
#include "boot_low_level_init.h"
#include "boot_delay.h"
#include "boot_clock.h"
#include "fdp.h"
#include "fdp_watchdog.h"
#include "fdp_jump.h"
#include "uart_dev.h"
#include "board_cfg.h"
#include "sysctl.h"
#include "bootrom_PIN.h"

BOOT_BSS bootromPIN_t *bootromPIN;

int main(void)
{
	fdp_record_stat_t *fdp_status = NULL;
	fdp_device_type_t fdp_type = FDP_DEV_UART;
	bootromPIN_t *bootromPIN1;

	boot_low_level_init(DEVICE_AHBCLK_FREQ);

	/* Enable peripheral. */
	SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_CAN);
	SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_SCI);
	SysCtl_resetCan();
	SysCtl_resetSci();
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CAN);
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SCI);

	boot_sysctl_clock_init(DEVICE_PLLCLK_FREQ, DEVICE_OSCSRC_FREQ, DEVICE_SYSCLK_DIV,
					DEVICE_AHBCLK_DIV, DEVICE_APBCLK_DIV, __HSE_ENABLE, __HSE_CLOCK_TYPE);

	boot_delay_init(DEVICE_SYSCLK_FREQ);

	bootromPIN = (bootromPIN_t *)0x1fff4500;
	bootromPIN1 = (bootromPIN_t *)(0x1fff4500 + bootromPIN->TABLE_SIZE);

	if (bootromPIN1->ENABLE_TABLE == BOOTROM_ENABLE_TABLE_KEY)
		bootromPIN = bootromPIN1;

reinit:
	if (fdp_init_all_device() != FDP_SUCCESS)
		goto reinit;

	while (fdp_wait_first_device_ack(&fdp_type) != FDP_SUCCESS);

	if (fdp_type != FDP_DEV_UART)
		fdp_device_deinit(FDP_DEV_UART);

	if (fdp_type != FDP_DEV_CAN_CTRL)
		fdp_device_deinit(FDP_DEV_CAN_CTRL);

	fdp_status = fdp_init(fdp_type);

	if (fdp_status == NULL)
		goto reinit;

	while (1) {
		fdp_receive_data_to_buffer();

		fdp_download(fdp_status->dev_type);

		if (fdp_status->fdp_done) {
			boot_delay_us(1000);
			goto reinit;
		}

		if (fdp_watchdog() == FDP_SUCCESS) {
			boot_delay_us(1000);
			goto reinit;
		}
	}

	return 0;
}
