/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "boot_hw_type.h"
#include "boot_low_level_init.h"
#include "boot_jump.h"
#include "boot_interrupt.h"
#include "boot_clock.h"
#include "boot_board_cfg.h"
#include "boot_sysctl_subcrg_peripheral.h"
#include "boot_gpio.h"
#include "boot_uart.h"
#include "boot_flash.h"
#include "boot_riscv_cache.h"
#include "pinmap.h"

BOOT_DATA uint32_t int_flag0 = 3;
BOOT_BSS uint32_t int_flag1;

BOOT_TEXT void boot_uart_irq_handler(void)
{
	uint32_t status;
	uint8_t rx_msg;

	status = boot_uart_get_interrupt_status(SCIA_BASE);

	if (status & BOOT_UART_INT_RXFF) {
		boot_uart_receive_msg_fifo(SCIA_BASE, &rx_msg);
		boot_uart_transmit_msg_fifo(SCIA_BASE, rx_msg);

		boot_uart_clear_interrupt_status(SCIA_BASE, BOOT_UART_INT_RXFF);
	}

	int_flag0++;
	int_flag1++;
}

extern "C" BOOT_TEXT int __boot_main(char argc, char *argv[])
{
	boot_flash_efc_status_t efc0_status;
	boot_flash_efc_status_t efc1_status;
	uint8_t buf[16];

	boot_low_level_init(BOOT_AHBCLK_FREQ);

	boot_sysctl_clock_init(BOOT_PLLCLK_FREQ, BOOT_OSCSRC_FREQ, BOOT_SYSCLK_DIV, BOOT_AHBCLK_DIV, BOOT_APBCLK_DIV, __BOOT_HSE_ENABLE, __BOOT_HSE_CLOCK_TYPE);

	boot_sysctl_enable_peripheral(BOOT_SYSCTL_PERIPH_CLK_SCI);

	/* UART initialize */
	boot_gpio_set_pin_config(GPIO_28_SCIA_RX);
	boot_gpio_set_pin_config(GPIO_29_SCIA_TX);

	boot_gpio_set_pad_config(28, BOOT_GPIO_PIN_TYPE_PULLUP);
	boot_gpio_set_pad_config(29, BOOT_GPIO_PIN_TYPE_PULLUP);

	boot_uart_reset(SCIA_BASE);
	boot_uart_set_config(SCIA_BASE, BOOT_APBCLK_FREQ, 115200,BOOT_UART_CONFIG_WLEN_8 |
						BOOT_UART_CONFIG_STOP_ONE | BOOT_UART_CONFIG_PAR_NONE);
	boot_uart_reset_channel(SCIA_BASE);

	boot_uart_set_fifo_level(SCIA_BASE, BOOT_UART_FIFO_TX0, BOOT_UART_FIFO_RX1);

	boot_uart_clear_interrupt_status(SCIA_BASE, BOOT_UART_INT_TXFF | BOOT_UART_INT_RXFF);

	boot_uart_enable_fifo_mode(SCIA_BASE);

	boot_uart_disable_interrupt(SCIA_BASE, 0xff);

	boot_uart_enable_interrupt(SCIA_BASE, BOOT_UART_INT_RXFF);

	boot_uart_enable_module(SCIA_BASE);

	/* register UART interrupt handler. */
	boot_interrupt_register(BOOT_INT_SCIA_RX, BOOT_ECLIC_LEVEL_TRIGGER, 0, 0, boot_uart_irq_handler);
	boot_interrupt_enable(BOOT_INT_SCIA_RX);

	boot_enable_irq();

	for (uint32_t i = 0; i < 16; i++)
		buf[i] = i;

	/* clear efc status */
	boot_flash_clear_efc_status(BOOT_FLASH_EFC0_ADDR, BOOT_FLASH_ALL_STATUS_MASK);
	boot_flash_clear_efc_status(BOOT_FLASH_EFC1_ADDR, BOOT_FLASH_ALL_STATUS_MASK);

	/* erase the flash */
	boot_flash_erase_main_flash_sector(0x08008000);
	boot_flash_erase_main_flash_sector(0x08040000);

	/* Wait for the flash erase to complete. */
	do {
		boot_flash_get_efc_status(BOOT_FLASH_EFC0_ADDR, &efc0_status);
		boot_flash_get_efc_status(BOOT_FLASH_EFC1_ADDR, &efc1_status);
	} while (efc0_status.efc_busy || efc1_status.efc_busy);

	boot_flash_clear_efc_status(BOOT_FLASH_EFC0_ADDR, BOOT_FLASH_ALL_STATUS_MASK);

	/* write data to flash */
	boot_flash_write_main_flash(0x08008000, buf);
	boot_machine_mode_invalid_DCache_lines(0x08008000, 1);

	/* Wait for the flash write to complete. */
	do {
		boot_flash_get_efc_status(BOOT_FLASH_EFC0_ADDR, &efc0_status);
	} while (efc0_status.efc_busy);

	/* clear efc status */
	boot_flash_clear_efc_status(BOOT_FLASH_EFC0_ADDR, BOOT_FLASH_ALL_STATUS_MASK);

	/* write data to flash */
	boot_flash_write_main_flash(0x08008020, buf);
	boot_machine_mode_invalid_DCache_lines(0x08008020, 1);

	/* Wait for the flash write to complete. */
	do {
		boot_flash_get_efc_status(BOOT_FLASH_EFC0_ADDR, &efc0_status);
	} while (efc0_status.efc_busy);

	/* clear efc status */
	boot_flash_clear_efc_status(BOOT_FLASH_EFC0_ADDR, BOOT_FLASH_ALL_STATUS_MASK);

	/* write data to flash */
	boot_flash_write_main_flash(0x08040000, buf);
	boot_machine_mode_invalid_DCache_lines(0x08040000, 1);

	/* Wait for the flash write to complete. */
	do {
		boot_flash_get_efc_status(BOOT_FLASH_EFC1_ADDR, &efc1_status);
	} while (efc1_status.efc_busy);

	/* clear efc status */
	boot_flash_clear_efc_status(BOOT_FLASH_EFC1_ADDR, BOOT_FLASH_ALL_STATUS_MASK);

	/* Jump to the first instruction position of the APP program. */
//	boot_jump_address(0x08008000U);

	while (1);

	return 0;
}
