/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __BOOT_INTERRUPT_H__
#define __BOOT_INTERRUPT_H__

#ifdef __cplusplus
extern "C"{
#endif

#include "boot_hw_type.h"
#include "boot_riscv_eclic.h"
#include "boot_riscv_cache.h"
#include "boot_irq_number.h"

typedef void (*boot_irq_handler_t)(void);

/**
 * @brief Initialize the ECLIC controller.
 *
 */
BOOT_TEXT void boot_interrupt_init(void);

/**
 * @brief  Set Interrupt Vector of a specific interrupt
 *
 * This function set interrupt handler address of the specific interrupt \em IRQn.
 * @param [in] IRQn Interrupt number
 * @param [in] vector Interrupt handler address
 *
 * - IRQn must not be negative.
 * - You can set the @ref CSR_CSR_MTVT to set interrupt vector table entry address.
 * - If your vector table is placed in readonly section, the vector for IRQn will not be modified.
 *   For this case, you need to use the correct irq handler name defined in your vector table as
 *   your irq handler function name.
 * - This function will only work correctly when the vector table is placed in an read-write enabled section.
 *
 * - @ref ECLIC_GetVector
 */
BOOT_TEXT void boot_interrupt_set_vector(Eclic_Reg_t *eclic, BOOT_IRQn_t IRQn, uintptr_t vector);

/**
 * @brief Register an interrupt handler function for a specific interrupt number.
 *
 * @param IRQn The interrupt number to register the handler for.
 * @param trigger_type The trigger type of the interrupt.
 * @param level The interrupt level.
 * @param priority The interrupt priority.
 * @param handler The interrupt handler function.
 */
BOOT_TEXT void boot_interrupt_register(BOOT_IRQn_t IRQn, BOOT_ECLIC_TRIGGER_t trigger_type, uint8_t level, uint8_t priority, boot_irq_handler_t handler);

/**
 * @brief Unregister an interrupt
 * @param [in] IRQn: External interrupt number.
 *
 * This parameter can be an enumerator of IRQn_Type enumeration
 *
 * @retval None
 */
BOOT_TEXT void boot_interrupt_unregister(BOOT_IRQn_t IRQn);

/**
 * @brief Enable an interrupt
 * @param [in] IRQn: interrupt number.
 *
 * This parameter can be an enumerator of IRQn_Type enumeration
 *
 * @retval None
 */
BOOT_TEXT void boot_interrupt_enable(BOOT_IRQn_t IRQn);

/**
 * @brief Disable an interrupt
 * @param [in] IRQn: interrupt number.
 *
 * This parameter can be an enumerator of IRQn_Type enumeration
 *
 * @retval None
 */
BOOT_TEXT void boot_interrupt_disable(BOOT_IRQn_t IRQn);

/**
 * @brief Enable IRQ Interrupts
 * @details Enables IRQ interrupts by setting the MIE-bit in the MSTATUS Register.
 * @remarks
 * Can only be executed in Privileged modes.
 */
BOOT_TEXT void boot_enable_irq(void);

/**
 * @brief Disable IRQ Interrupts
 * @details Disables IRQ interrupts by clearing the MIE-bit in the MSTATUS Register.
 * @remarks
 * Can only be executed in Privileged modes.
 */
BOOT_TEXT void boot_disable_irq(void);

#ifdef __cplusplus
}
#endif

#endif
