/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __BOOT_SYSCTL_SUCRG_PERIPHERAL_V3_0_H__

#ifdef __cplusplus
extern "C"{
#endif

#if (GS32F00xx == 0x3000)

#include "boot_hw_type.h"
#include "inc/hw_memmap.h"
#include "inc/hw_sysctl_ahb.h"

#define BOOT_CRG_CFG_ADDR_SHIFT		25
#define BOOT_CRG_CFG_ADDR_MASK		0x7F
#define BOOT_CRG_CFG_MASK_SHIFT		5
#define BOOT_CRG_CFG_MASK_MASK		0xFFFFF
#define BOOT_CRG_CFG_OFF_SHIFT		0
#define BOOT_CRG_CFG_OFF_MASK		0x1F

#define BOOT_GS32F00xx_CRG_CFG(addr, mask, off)		((addr) << BOOT_CRG_CFG_ADDR_SHIFT | (mask) << BOOT_CRG_CFG_MASK_SHIFT | (off))

/**
 *
 * The following are values that can be passed to SysCtl_enablePeripheral()
 * and SysCtl_disablePeripheral() as the  peripheral parameter.
 *
 */
typedef enum
{
	/* offset == 0x08 */
	BOOT_SYSCTL_PERIPH_CLK_TIMER	= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 0),
	BOOT_SYSCTL_PERIPH_CLK_SCI		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x07, 4),
	BOOT_SYSCTL_PERIPH_CLK_SCIA		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 4),
	BOOT_SYSCTL_PERIPH_CLK_SCIB		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 5),
	BOOT_SYSCTL_PERIPH_CLK_SCIC		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 6),
	BOOT_SYSCTL_PERIPH_CLK_I2C		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x03, 8),
	BOOT_SYSCTL_PERIPH_CLK_I2CA		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 8),
	BOOT_SYSCTL_PERIPH_CLK_I2CB		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 9),
	BOOT_SYSCTL_PERIPH_CLK_PMBUS	= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 10),
	BOOT_SYSCTL_PERIPH_CLK_SPI		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x03, 12),
	BOOT_SYSCTL_PERIPH_CLK_SPIA		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 12),
	BOOT_SYSCTL_PERIPH_CLK_SPIB		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 13),
	BOOT_SYSCTL_PERIPH_CLK_CAN		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x03, 16),
	BOOT_SYSCTL_PERIPH_CLK_CANA		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 16),
	BOOT_SYSCTL_PERIPH_CLK_CANB		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 17),
	BOOT_SYSCTL_PERIPH_CLK_LIN		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x03, 20),
	BOOT_SYSCTL_PERIPH_CLK_LINA		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 20),
	BOOT_SYSCTL_PERIPH_CLK_LINB		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 21),
	BOOT_SYSCTL_PERIPH_CLK_EXMC		= BOOT_GS32F00xx_CRG_CFG(0x8, 0x01, 23),
	/* offset == 0x0C */
	BOOT_SYSCTL_PERIPH_CLK_CRC		= BOOT_GS32F00xx_CRG_CFG(0xC, 0x01, 4),
	BOOT_SYSCTL_PERIPH_CLK_EPG		= BOOT_GS32F00xx_CRG_CFG(0xC, 0x01, 12),
	BOOT_SYSCTL_PERIPH_CLK_ERAD		= BOOT_GS32F00xx_CRG_CFG(0xC, 0x01, 16),
	BOOT_SYSCTL_PERIPH_CLK_XBAR		= BOOT_GS32F00xx_CRG_CFG(0xC, 0x01, 20),
	BOOT_SYSCTL_PERIPH_CLK_AES		= BOOT_GS32F00xx_CRG_CFG(0xC, 0x01, 22),
	/*offset == 0x10*/
	BOOT_SYSCTL_PERIPH_CLK_CLB		= BOOT_GS32F00xx_CRG_CFG(0x10, 0x07, 0),
	BOOT_SYSCTL_PERIPH_CLK_CLB1		= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 0),
	BOOT_SYSCTL_PERIPH_CLK_CLB2		= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 1),
	BOOT_SYSCTL_PERIPH_CLK_CLBCOM	= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 2),
	BOOT_SYSCTL_PERIPH_CLK_SDFM		= BOOT_GS32F00xx_CRG_CFG(0x10, 0x07, 8),
	BOOT_SYSCTL_PERIPH_CLK_SDFM1	= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 8),
	BOOT_SYSCTL_PERIPH_CLK_SDFM2	= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 9),
	BOOT_SYSCTL_PERIPH_CLK_SDFMCOM	= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 10),
	BOOT_SYSCTL_PERIPH_CLK_EQEP		= BOOT_GS32F00xx_CRG_CFG(0x10, 0x07, 12),
	BOOT_SYSCTL_PERIPH_CLK_EQEP1	= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 12),
	BOOT_SYSCTL_PERIPH_CLK_EQEP2	= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 13),
	BOOT_SYSCTL_PERIPH_CLK_EQEP3	= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 14),
	BOOT_SYSCTL_PERIPH_CLK_CLB_ENC	= BOOT_GS32F00xx_CRG_CFG(0x10, 0x03, 16),
	BOOT_SYSCTL_PERIPH_CLK_CLB_ENC1	= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 16),
	BOOT_SYSCTL_PERIPH_CLK_CLB_ENC2	= BOOT_GS32F00xx_CRG_CFG(0x10, 0x01, 17),
	/*offset == 0x14*/
	BOOT_SYSCTL_PERIPH_CLK_EPWM		= BOOT_GS32F00xx_CRG_CFG(0x14, 0x1FF, 0),
	BOOT_SYSCTL_PERIPH_CLK_EPWM1	= BOOT_GS32F00xx_CRG_CFG(0x14, 0x1, 0),
	BOOT_SYSCTL_PERIPH_CLK_EPWM2	= BOOT_GS32F00xx_CRG_CFG(0x14, 0x1, 1),
	BOOT_SYSCTL_PERIPH_CLK_EPWM3	= BOOT_GS32F00xx_CRG_CFG(0x14, 0x1, 2),
	BOOT_SYSCTL_PERIPH_CLK_EPWM4	= BOOT_GS32F00xx_CRG_CFG(0x14, 0x1, 3),
	BOOT_SYSCTL_PERIPH_CLK_EPWM5	= BOOT_GS32F00xx_CRG_CFG(0x14, 0x1, 4),
	BOOT_SYSCTL_PERIPH_CLK_EPWM6	= BOOT_GS32F00xx_CRG_CFG(0x14, 0x1, 5),
	BOOT_SYSCTL_PERIPH_CLK_EPWM7	= BOOT_GS32F00xx_CRG_CFG(0x14, 0x1, 6),
	BOOT_SYSCTL_PERIPH_CLK_EPWM8	= BOOT_GS32F00xx_CRG_CFG(0x14, 0x1, 7),
	BOOT_SYSCTL_PERIPH_CLK_EPWMCOM	= BOOT_GS32F00xx_CRG_CFG(0x14, 0x1, 8),
	/*offset == 0x18*/
	BOOT_SYSCTL_PERIPH_CLK_ECAP		= BOOT_GS32F00xx_CRG_CFG(0x18, 0x07, 0),
	BOOT_SYSCTL_PERIPH_CLK_ECAP1	= BOOT_GS32F00xx_CRG_CFG(0x18, 0x01, 0),
	BOOT_SYSCTL_PERIPH_CLK_ECAP2	= BOOT_GS32F00xx_CRG_CFG(0x18, 0x01, 1),
	BOOT_SYSCTL_PERIPH_CLK_ECAP3	= BOOT_GS32F00xx_CRG_CFG(0x18, 0x01, 2),
	/*offset == 0x1C*/
	BOOT_SYSCTL_PERIPH_CLK_CMPSS	= BOOT_GS32F00xx_CRG_CFG(0x1C, 0xF, 0),
	BOOT_SYSCTL_PERIPH_CLK_CMPSS1	= BOOT_GS32F00xx_CRG_CFG(0x1C, 0x1, 0),
	BOOT_SYSCTL_PERIPH_CLK_CMPSS2	= BOOT_GS32F00xx_CRG_CFG(0x1C, 0x1, 1),
	BOOT_SYSCTL_PERIPH_CLK_CMPSS3	= BOOT_GS32F00xx_CRG_CFG(0x1C, 0x1, 2),
	BOOT_SYSCTL_PERIPH_CLK_CMPSS4	= BOOT_GS32F00xx_CRG_CFG(0x1C, 0x1, 3),
	/*offset == 0x24*/
	BOOT_SYSCTL_PERIPH_CLK_ADC		= BOOT_GS32F00xx_CRG_CFG(0x24, 0x7, 0),
	BOOT_SYSCTL_PERIPH_CLK_ADCA		= BOOT_GS32F00xx_CRG_CFG(0x24, 0x1, 0),
	BOOT_SYSCTL_PERIPH_CLK_ADCC		= BOOT_GS32F00xx_CRG_CFG(0x24, 0x1, 1),
	BOOT_SYSCTL_PERIPH_CLK_ADCCOM	= BOOT_GS32F00xx_CRG_CFG(0x24, 0x1, 2),

	BOOT_SYSCTL_PERIPH_CLK_TBCLKSYNC	= 0xFFFFFFFF
} Boot_SysCtl_PeripheralPCLOCKCR_t;

/**
 * @brief Enables the clock to a peripheral.
 *
 * @param [in] peripheral is the peripheral to enable, see type Boot_SysCtl_PeripheralPCLOCKCR_t.
 *
 * Peripherals are enabled with this function.  At power-up, all peripherals
 * are disabled; they must be enabled in order to operate or respond to
 * register reads/writes.
 *
 * @remarks
 * Note that there should be at least 5 cycles delay between enabling the
 * peripheral clock and accessing the peripheral registers. The delay should be
 * added by the user if the peripheral is accessed immediately after this
 * function call.
 * @return None
 */
BOOT_TEXT void boot_sysctl_enable_peripheral(Boot_SysCtl_PeripheralPCLOCKCR_t peripheral);

/**
 * @brief Disables a peripheral.
 *
 * @param [in]   peripheral is the peripheral to disable.
 *
 * Peripherals are disabled with this function.  Once disabled, they will not
 * operate or respond to register reads/writes. But the peripheral registers are not reset.
 *
 * @return  None
 */
BOOT_TEXT void boot_sysctl_disable_peripheral(Boot_SysCtl_PeripheralPCLOCKCR_t peripheral);

#endif

#ifdef __cplusplus
}
#endif

#endif
