/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "boot_check_algorithm.h"

uint8_t boot_algorithm_checksum8(uint8_t *p_data, uint32_t len)
{
	uint8_t sum = 0;

	for (uint32_t i = 0; i < len; i++) {
		sum += p_data[i];
	}

	return sum;
}

/**
 * @brief CRC16 modbus verification algorithm
 *
 * @param data Data buffer.
 * @param length Data length.
 * @return uint16_t Return the CRC16 modbus verification value.
 */
uint16_t boot_crc16_modbus(uint8_t *data, uint32_t length)
{
	uint16_t crc = 0xffff;
	/* Polynomial: x^16 + x^15 + x^2 + 1 */
	uint16_t polynomial = 0xa001;

	for (uint32_t i = 0; i < length; i++) {
	crc ^= data[i];

		for (uint8_t j = 0; j < 8; j++) {
			if (crc & 0x0001)
				crc = (crc >> 1) ^ polynomial;
			else
				crc >>= 1;
		}
	}

	return crc;
}

/**
 * @brief CRC32 verification algorithm
 *
 * @param data Data buffer.
 * @param length Data length.
 * @return uint32_t Return the CRC32 verification value.
 */
uint32_t boot_crc32(uint8_t *data, uint32_t length)
{
	uint8_t i;
	/* Initial value */
	uint32_t crc = 0xffffffff;

	while(length--) {
		crc ^= *data++;
		for (i = 0; i < 8; ++i) {
			if (crc & 1)
				/* 0xEDB88320= reverse 0x04C11DB7 */
				crc = (crc >> 1) ^ 0xEDB88320;
			else
				crc = (crc >> 1);
		}
	}

	return ~crc;
}
