/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#if (GS32F00xx == 0x3000)

#include "boot_hw_type.h"
#include "boot_clock_v3_0.h"
#include "boot_delay.h"
#include "boot_gpio.h"
#include "boot_sysctl.h"

BOOT_TEXT void boot_Auto_setupXtal(uint32_t osc_freq, uint8_t hse_type)
{
	uint32_t clkRefPinCtrl = 0;
	uint32_t clk_count;

	/* disable HSE monitor */
	boot_sysctl_set_hse_cmd_enable(0x0);

	if (osc_freq <= 4*1000*1000)					/* 1 ~ 4MHz */
		clkRefPinCtrl = GPIO_ClkRef_1MHz_TO_4MHz;
	else if (osc_freq <= 12*1000*1000)				/* 4.1 ~ 12MHz */
		clkRefPinCtrl = GPIO_ClkRef_4MHz_TO_12MHz;
	else if (osc_freq <= 24*1000*1000)				/* 12.1 ~ 24MHz */
		clkRefPinCtrl = GPIO_ClkRef_12MHz_TO_24MHz;
	else											/* 24.1 ~ 48MHz */
		clkRefPinCtrl = GPIO_ClkRef_24MHz_TO_48MHz;

	clkRefPinCtrl |= GPIO_ClkRef_R_500K;

	if (hse_type) {									/* external oscillator */
		clkRefPinCtrl |= GPIO_ClkRef_OSCILLATOR;
		boot_gpio_set_clk_ref(clkRefPinCtrl);
	} else {										/* external crystal */
		clkRefPinCtrl |= GPIO_ClkRef_CRYSTAL;
		boot_gpio_set_clk_ref(clkRefPinCtrl);
	}

	/* enable HSE monitor */
	boot_sysctl_set_hse_cmd_enable(0x1);

	boot_sysctl_set_ext_ref_clk_sel(EXT_REF_CLK_SEL_XTAL);
	boot_sysctl_set_ref_clk_sel(REF_CLK_TYPE_EXTERNAL_XTAL);

	boot_delay_tick(2000);

	while (1) {
		boot_sysctl_set_x1_cnt_clr();

		if (boot_sysctl_get_cmd_check_fail_rpt()) {
			boot_sysctl_set_cmd_st_clr(1);
			boot_sysctl_set_cmd_st_clr(0);

			boot_delay_tick(2000);
		} else {
			clk_count = boot_sysctl_get_x1_cnt();
			if (clk_count >= 20)
				break;
		}
	}
}

BOOT_TEXT void boot_sysctl_clock_init(uint32_t nPllClkFreq, uint32_t nOscFreq, uint32_t nSysClkDiv,
							uint32_t nAhbClkDiv, uint32_t nApbClkDiv, bool hse_en, uint8_t hse_type)
{
	/* --- Step 0 : decrease APB clock the half of the normal frequency */
	boot_sysctl_set_apb_clock_div(CRG_DIV_8);

	boot_sysctl_set_dsp_sysclk_div(CRG_DIV_4);

	boot_sysctl_set_periclk_div(CRG_DIV_8);

	boot_sysctl_set_sys_div_Load();

	boot_delay_tick(150 * 10);

	boot_sysctl_set_sysclk_s2n_enable(0);

	boot_delay_tick(150 * 10);

	boot_sysctl_set_ref_clk_sel(REF_CLK_TYPE_INTERNAL_OSC2_10MHZ);
	boot_delay_tick(2000);

	while (1) {
		if (boot_sysctl_get_cmd_check_fail_rpt()) {
			boot_sysctl_set_cmd_st_clr(1);
			boot_sysctl_set_cmd_st_clr(0);
			boot_delay_tick(2000);
		} else
			break;
	}

	/**
	 * --- Step 1 : If(HSE_ENABLE) Config HSE_OSC and Clock Detector,
	 * Config Clock Detector, ELSE Read Eflash rpt, Config HSI TRIM
	 */
	if (hse_en == 1)
		boot_Auto_setupXtal(nOscFreq, hse_type);

	/* --- Step 2 : Config the PLL, Read PLL_LCK until PLL Locked */
	boot_sysctl_set_pll_disable_sscg(0x01);
	boot_sysctl_set_pll_reset_ptr(0x01);
	boot_sysctl_set_pll_pd(0x01);

	boot_delay_tick(150 * 10);

	/* FOUT = (FREF * FBDIV) / (REFDIV * POSTDIV1 * POSTDIV2) */
	/* FOUT = (FREF / REFDIV) * ((FBDIV + FRAC/2^24) / (POSTDIV1 * POSTDIV2)) */

	/* pll_clk = 2 * sysclk, adc_clk = pll_clk/2 */
	boot_sysctl_set_pll_fb_div_cfg(2 * (nPllClkFreq) / nOscFreq);
	boot_sysctl_set_pll_frac_cfg(0);

	/* [2:0]:POSTDIV1 [5:3]:POSTDIV2 [11:6]:REFDIV */
	boot_sysctl_set_pll_div_cfg(0x01 | (0x01 << 3) | (0x01 << 6));
	/* Modulation Freq = F_clksscg / (DIVVAL * 128) */
	boot_sysctl_set_pll_div_val(0x00000001UL);
	/* 5'd1:1% ... 5'd31:31% */
	boot_sysctl_set_pll_spread(0);

	/* 0:center-spread 1:downspread */
	boot_sysctl_set_pll_down_spread(0);
	boot_sysctl_set_pll_othpd_cfg(0);

	boot_delay_tick(10 * 10);

	boot_sysctl_set_pll_disable_sscg(0);
	boot_delay_tick(10 * 10);

	boot_sysctl_set_pll_pd(0);
	/* wait 150us+ for PLL locking, TODO: delay time for 240MHz */
	boot_delay_tick(150 * 10);
	boot_sysctl_set_pll_reset_ptr(0);

	/* Wait until PLL lock status is 1 */
	while(boot_sysctl_get_pll_lock() == 0);

	boot_sysctl_clear_pll_unlck_his();

	/**
	 * --- Step 3 : SYSCLK Slow2Normal,
	 * Config SYSCLK Divider to 2div and then to 1div
	 */
	boot_sysctl_set_sysclk_s2n_enable(0x01);
	boot_delay_tick(150 * 10);

	/* --- Step 4 : Config APBCLK/CANCLK/ANACLK Divider */
	boot_sysctl_set_dsp_sysclk_div(nSysClkDiv);

	boot_sysctl_set_periclk_div(nAhbClkDiv);
	boot_sysctl_set_apb_clock_div(nApbClkDiv);

	/**
	 * After writing 1 to the register CFG_SYS_DIV_LOAD_POS,
	 * cfg_div_dsp_sys, cfg_div_apb_sys registers are updated,
	 * DSP/APB divider are updated at the same time
	 */
	boot_sysctl_set_sys_div_Load();
}

#endif		/* GS32F00xx */
