/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <time.h>
#include <stdlib.h>

#include "device.h"
#include "interrupt.h"
#include "fdp.h"

/* Global variable */
volatile uint32_t task10msFlag = 0;

extern uint32_t FLASH_APP1_ADDR;
extern uint32_t FLASH_CORE0_REGION_ADDR;

__INTERRUPT void APB_TIMER1_IRQ_HANDLER(void)
{
	task10msFlag = 1U;
	GPIO_togglePin(GPIO_PIN_LED1);
	CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
	__DSB();
}

int32_t Timer1_init(uint32_t period)
{
	volatile uint32_t cpuTIntPending = 0U;

	/* Initialize CPUTimer with interrupt enable */
	CPUTimer_init(CPUTIMER1_BASE, period-1);

	/* For CPUTIMER1 used both in Boot loader (CPU1) and  APP2 (CPU2, it need to clear pending */
	cpuTIntPending = __ECLIC_GetPendingIRQ(INT_TIMER1);
	if (cpuTIntPending != 0)
		__ECLIC_ClearPendingIRQ(INT_TIMER1);

	Interrupt_register(INT_TIMER1, &APB_TIMER1_IRQ_HANDLER);
	Interrupt_enable(INT_TIMER1);

	return 0;
}

void cInit_Leds(void)
{
	GPIO_setAnalogMode(GPIO_PIN_LED1, GPIO_ANALOG_DISABLED);
	GPIO_setAnalogMode(GPIO_PIN_LED2, GPIO_ANALOG_DISABLED);
    GPIO_setPinConfig(GPIO_CFG_LED1);
    GPIO_setPinConfig(GPIO_CFG_LED2);
    GPIO_setPadConfig(GPIO_PIN_LED1, GPIO_PIN_TYPE_STD);
    GPIO_setPadConfig(GPIO_PIN_LED2, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(GPIO_PIN_LED1, GPIO_DIR_MODE_OUT);
    GPIO_setDirectionMode(GPIO_PIN_LED2, GPIO_DIR_MODE_OUT);
    GPIO_WritePin(GPIO_PIN_LED1, 0);
    GPIO_WritePin(GPIO_PIN_LED2, 1);
}

int main(void)
{
	fdp_device_type_t fdp_type;
	fdp_record_stat_t *fdp_status;
	fdp_app_image_info_t *info;
	uint32_t tick500ms = 0U;

#if defined (MONITOR_PACKAGE)&& MONITOR_PACKAGE
	CPU_clearCycleCnt();
#endif
	__disable_irq();

	Device_init();

	/* Initial Led */
	cInit_Leds();

	/* Check APP image and jump to APP. */
	if (!fdp_is_entry_load_mode()) {
		/* Check if the image is valid. */
		if (!fdp_check_jtag_app((uintptr_t)&FLASH_APP1_ADDR) || !fdp_check_image_base_on_info((uintptr_t)&FLASH_APP1_ADDR))
		{
			/* Jump to the address of the first instruction in the APP program. */
			info = (fdp_app_image_info_t *)&FLASH_APP1_ADDR;
			fdp_set_entry_load_mode(false);
			fdp_jump_to_address(info->app_start_addr);
		}
	}
	Timer1_init(DEVICE_APBCLK_FREQ / 100);	//10ms

update_image :

	fdp_set_entry_load_mode(false);

	if (fdp_boot_sel_mode_init())
		goto update_image;

	if (fdp_boot_mode_get(&fdp_type))
		goto update_image;

	fdp_status = fdp_init(fdp_type);

	if (fdp_status == NULL)
		goto update_image;

	__enable_irq();

	while (1) {
		fdp_download(fdp_status->dev_type);

		if ((fdp_status->release_core ==1U) && (fdp_status->image_ready == true)) {
			if (fdp_check_image_info((uintptr_t)&FLASH_APP1_ADDR) == FDP_SUCCESS) {
				__disable_irq();
				Interrupt_disable(INT_TIMER1);
				DELAY_US(5000);
				//switch boot mode for next flow
				SysCtl_setSrstN(1);
				while(1);
			}
		}

		/* fix the issue that the counter out quickly because the DSP run too fast when FlashILM configuration. */
		if (task10msFlag == 1U)
		{
			fdp_watchdog();
			task10msFlag = 0U;
			if (tick500ms++%50 == 0U)
				GPIO_togglePin(GPIO_PIN_LED2);
		}
	}

	goto update_image;

	return 0;
}
