/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <string.h>

#include "fdp_err.h"
#include "fdp.h"

#include "device.h"
#include "m_can_dev.h"
#include "mcan.h"

#define M_CAN_BASE		CANA_BASE
#define M_CAN_IRQ_NUM	INT_CANA1
#define M_CAN_IRQ_NUM1	INT_CANA0

#if ((GS32_PART_NUM==0x0025)||(GS32_PART_NUM==0x0039A)||(GS32_PART_NUM==0x0049A))
#define CAN_TX_PIN	GPIO_32_CANA_TX
#define CAN_RX_PIN	GPIO_33_CANA_RX
#define CAN_TX_PIN_NUM	32
#define CAN_RX_PIN_NUM	33
#endif

#define M_CAN_TX_ID		0x456
#define M_CAN_RX_ID		0x123

#define M_CAN_MSG_MAX_LEN				MCAN_DATA_LENGTH_8

#define M_CAN_MSG_MAX_LENGTH			MCAN_ELEM_SIZE_64BYTES
#define M_CAN_MSG_BUFFER_SIZE			(MCAN_getMsgObjSize(M_CAN_MSG_MAX_LENGTH) + 12U)

#define M_CAN_RAM_BASE_ADDR				0U
#define M_CAN_STANDARD_FILTER_BASE_ADDR	M_CAN_RAM_BASE_ADDR
#define M_CAN_STANDARD_FILTER_NUM		(4U)
#define M_CAN_STANDARD_FILTER_SIZE		(4U)

/* Note: The size of all memory regions cannot exceed 4096 bytes. */
#define M_CAN_EXTENDED_FILTER_BASE_ADDR	(M_CAN_STANDARD_FILTER_BASE_ADDR + \
										(M_CAN_STANDARD_FILTER_NUM * \
										 M_CAN_STANDARD_FILTER_SIZE))
#define M_CAN_EXTENDED_FILTER_NUM		(1U)
#define M_CAN_EXTENDED_FILTER_SIZE		(8U)

#define M_CAN_RXFIFO0_BASE_ADDR			(M_CAN_EXTENDED_FILTER_BASE_ADDR + \
										(M_CAN_EXTENDED_FILTER_NUM * \
										 M_CAN_EXTENDED_FILTER_SIZE))
#define M_CAN_RXFIFO0_NUM				(5U)
#define M_CAN_RXFIFO0_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_RXFIFO1_BASE_ADDR			(M_CAN_RXFIFO0_BASE_ADDR + \
										(M_CAN_RXFIFO0_NUM * \
										 M_CAN_RXFIFO0_SIZE))
#define M_CAN_RXFIFO1_NUM				(5U)
#define M_CAN_RXFIFO1_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_RXBUFFER_BASE_ADDR		(M_CAN_RXFIFO1_BASE_ADDR + \
										(M_CAN_RXFIFO1_NUM * \
										 M_CAN_RXFIFO1_SIZE))
#define M_CAN_RXBUFFER_NUM				(8U)
#define M_CAN_RXBUFFER_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_TXEVENTFIFO_BASE_ADDR		(M_CAN_RXBUFFER_BASE_ADDR + \
										(M_CAN_RXBUFFER_NUM * \
										 M_CAN_RXBUFFER_SIZE))
#define M_CAN_TXEVENTFIFO_NUM			(10U)
#define M_CAN_TXEVENTFIFO_SIZE			(M_CAN_TX_EVENT_FIFO_SIZE)

#define M_CAN_TXBUFFER_BASE_ADDR		(M_CAN_TXEVENTFIFO_BASE_ADDR + \
										(M_CAN_TXEVENTFIFO_NUM * \
										 M_CAN_TXEVENTFIFO_SIZE))
#define M_CAN_TXBUFFER_NUM				(4U)
#define M_CAN_TXBUFFER_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_TXFIFOQUEUE_BASE_ADDR		(M_CAN_TXBUFFER_BASE_ADDR + \
										(M_CAN_TXBUFFER_NUM * \
										 M_CAN_TXBUFFER_SIZE))
#define M_CAN_TXFIFOQUEUE_NUM			(6U)
#define M_CAN_TXFIFOQUEUE_SIZE			(M_CAN_MSG_BUFFER_SIZE)

MCAN_RxNewDataStatus_t new_dat;
MCAN_RxMessage_t RxMsgBuf = {0};

__INTERRUPT void M_CAN_interrupt_irq_handler(void)
{
	uint32_t status;

	status = MCAN_getIntrStatus(M_CAN_BASE);

	if (status & MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF) {
		fdp_receive_data_to_buffer();
	}

	MCAN_getNewDataStatus(M_CAN_BASE, &new_dat);
	if (new_dat.statusHigh == 0 && new_dat.statusLow == 0)
		MCAN_clearIntrStatus(M_CAN_BASE, MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF);
}

static void M_CAN_config(void)
{
	MCAN_InitParams_t init_param;
	MCAN_ConfigParams_t configParams;
	MCAN_MsgRAMConfigParams_t RAMConfig;
	MCAN_BitTimingParams_t configBitrate;
	MCAN_StdMsgIDFilterElement_t std_filter;
	MCAN_ExtMsgIDFilterElement_t ext_filter;

	/* CAN mode configuration. */
	init_param.fdMode = false;
	init_param.fdFormat = MCAN_FD_ISO_11898_1;
	init_param.brsEnable = false;
	init_param.txpEnable = false;
	init_param.efbi = false;
	init_param.pxhddisable = false;
	/* Enable the auto retransmission. */
	init_param.darEnable = true;
	init_param.wkupReqEnable = false;
	init_param.autoWkupEnable = false;
	init_param.emulationEnable = false;
	init_param.wdcPreload = false;
	init_param.wmMarker = MCAN_WMM_8BIT_MODE;
	init_param.tdcEnable = false;
	init_param.tdcConfig.tdcf = 0U;
	init_param.tdcConfig.tdco = 0U;

	/* Set CAN extended feature. */
	configParams.asmEnable = false;
	configParams.monEnable = false;
	configParams.timeoutCntEnable = false;
	configParams.timeoutPreload = 1;
	configParams.timeoutSelect = MCAN_TIMEOUT_SELECT_CONT;

	configParams.tsClock = MCAN_INTERNAL_TIMESTAMP;
	configParams.tsSelect = MCAN_TSCNTVAL_ALWAYS0;
	configParams.tsPrescalar = 1;

	/* Set global configuration. */
	configParams.filterConfig.rrfs = true;
	configParams.filterConfig.rrfe = true;
	configParams.filterConfig.anfs = MCAN_NON_MATCH_REJECT_ACCEPT;
	configParams.filterConfig.anfe = MCAN_NON_MATCH_REJECT_ACCEPT;

	/* Set standard ID filter. */
	RAMConfig.lss = M_CAN_STANDARD_FILTER_NUM;
	RAMConfig.flssa = M_CAN_STANDARD_FILTER_BASE_ADDR;

	/* Set extended ID filter. */
	RAMConfig.lse = M_CAN_EXTENDED_FILTER_NUM;
	RAMConfig.flesa = M_CAN_EXTENDED_FILTER_BASE_ADDR;

	/* Set Rx fifo 0. */
	RAMConfig.rxFIFO0OpMode = MCAN_RXFIFO_OP_MODE_BLOCKING;
	RAMConfig.rxFIFO0startAddr = M_CAN_RXFIFO0_BASE_ADDR;
	RAMConfig.rxFIFO0size = M_CAN_RXFIFO0_NUM;
	RAMConfig.rxFIFO0waterMark = 0;
	RAMConfig.rxFIFO0ElemSize  = M_CAN_MSG_MAX_LENGTH;

	/* Set Rx fifo 1. */
	RAMConfig.rxFIFO1OpMode = MCAN_RXFIFO_OP_MODE_BLOCKING;
	RAMConfig.rxFIFO1startAddr = M_CAN_RXFIFO1_BASE_ADDR;
	RAMConfig.rxFIFO1size = M_CAN_RXFIFO1_NUM;
	RAMConfig.rxFIFO1waterMark = 0;
	RAMConfig.rxFIFO1ElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Rx buffer. */
	RAMConfig.rxBufStartAddr = M_CAN_RXBUFFER_BASE_ADDR;
	RAMConfig.rxBufElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Tx buffer */
	RAMConfig.txBufMode = MCAN_TXBUF_OP_IN_FIFO_MODE;
	RAMConfig.txStartAddr = M_CAN_TXBUFFER_BASE_ADDR;
	RAMConfig.txFIFOSize = M_CAN_TXFIFOQUEUE_NUM;
	RAMConfig.txBufNum = M_CAN_TXBUFFER_NUM;
	RAMConfig.txBufElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Tx event fifo. */
	RAMConfig.txEventFIFOStartAddr = M_CAN_TXEVENTFIFO_BASE_ADDR;
	RAMConfig.txEventFIFOSize = M_CAN_TXEVENTFIFO_NUM;
	RAMConfig.txEventFIFOWaterMark = 1;

	/* Input clock source 40Mhz, normal phase bitrate 500kbps. */
	configBitrate.nomRatePrescalar = 1;
	configBitrate.nomTimeSeg1 = 30;
	configBitrate.nomTimeSeg2 = 7;
	configBitrate.nomSynchJumpWidth = 1;

	/* Reset the MCAN module. */
	while (MCAN_isInReset(M_CAN_BASE));

	/* Check if the MCAN RAM is ready. */
	while (!MCAN_isMemInitDone(M_CAN_BASE));

	/* Set the MCAN mode to init mode. */
	if (!MCAN_setOpMode(M_CAN_BASE, MCAN_OPERATION_SW_INIT_MODE))
		while (1);

	/* Initialize the MCAN. */
	if (!MCAN_init(M_CAN_BASE, &init_param))
		while (1);

	if (!MCAN_config(M_CAN_BASE, &configParams))
		while (1);

	if (!MCAN_setBitTime(M_CAN_BASE, &configBitrate))
		while (1);

	if (!MCAN_msgRAMConfig(M_CAN_BASE, &RAMConfig))
		while (1);

	/* Add a new standard ID filter(Use Rx buffer 0). */
	std_filter.sfid1 = 0x123;
	std_filter.sfid2 = 0x0;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG;
	std_filter.sft = MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2;
	if (!MCAN_addStdMsgIDFilter(M_CAN_BASE, 0, &std_filter))
		while (1);

	/* Add a new standard ID filter(Use Rx buffer 1). */
	std_filter.sfid1 = 0x234;
	std_filter.sfid2 = 0x1;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG;
	std_filter.sft = MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2;
	if (!MCAN_addStdMsgIDFilter(M_CAN_BASE, 1, &std_filter))
		while (1);

	/* Add a new standard ID filter(Use Rx buffer 2). */
	std_filter.sfid1 = 0x345;
	std_filter.sfid2 = 0x2;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG;
	std_filter.sft = MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2;
	if (!MCAN_addStdMsgIDFilter(M_CAN_BASE, 2, &std_filter))
		while (1);

	/* Add a new extended ID filter(Use Rx fifo 0). */
	ext_filter.efid1 = 0x166666;
	ext_filter.efid2 = 0x1fffff;
	ext_filter.esync = MCAN_EXTIDF_SYNC_MSG_DISABLE;
	ext_filter.efec = MCAN_EXTIDF_ELE_PRIO_STORE_IN_FO0_OF_MATCH_ID;
	ext_filter.eft = MCAN_EXTIDF_RANGE_FROM_EFID1_TO_EFID2;
	if (!MCAN_addExtMsgIDFilter(M_CAN_BASE, 0, &ext_filter))
		while (1);

	/* The received extension frame will be subjected to AND operation with Ext_mask. */
	if (!MCAN_setExtIDAndMask(M_CAN_BASE, 0x166666))
		while (1);

	/* Enable/Disable Loopback mode. */
	if (!MCAN_lpbkModeEnable(M_CAN_BASE, MCAN_LPBK_MODE_INTERNAL, false))
		while (1);

	/* Enable MCAN. */
	if (!MCAN_setOpMode(M_CAN_BASE, MCAN_OPERATION_NORMAL_MODE))
		while (1);

	 /* Configuration the external timestamp clock source. */
	 if (!MCAN_extTSCounterConfig(M_CAN_BASE, 0xffff))
	 	while (1);

	 /* Enable/Disable external timestamp clock source. */
	 if (!MCAN_extTSCounterEnable(M_CAN_BASE, true))
	 	while (1);

	 /* Enable/Disable external timestamp overflow interrupt. */
	 if (!MCAN_extTSEnableIntr(M_CAN_BASE, false))
	 	while (1);

	/* Select MCAN interrupt route to interrupt line 0 or 1. */
	if (!MCAN_selectIntrLine(M_CAN_BASE, MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF |
									   MCAN_INT_SRC_RxFIFO0_NEW_MSG,
									   MCAN_INTERRUPT_LINE_0))
		while (1);

	/* Enable MCAN interrupt. */
	if (!MCAN_enableIntr(M_CAN_BASE, MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF |
								   MCAN_INT_SRC_RxFIFO0_NEW_MSG, true))
		while (1);

	if (!MCAN_enableIntrLine(M_CAN_BASE, MCAN_INTERRUPT_LINE_0, true))
		while (1);

	if (!MCAN_enableIntrLine(M_CAN_BASE, MCAN_INTERRUPT_LINE_1, true))
		while (1);
}

int M_CAN_Init(void)
{
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CAN);
	SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);
	SysCtl_setCanClkDiv(DEVICE_PLLCLK_FREQ / 40000000U);
	SysCtl_resetCana();
	SysCtl_resetCanb();

	GPIO_setPinConfig(CAN_TX_PIN);
	GPIO_setPinConfig(CAN_RX_PIN);
	GPIO_setPadConfig(CAN_TX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
	GPIO_setPadConfig(CAN_RX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);

	M_CAN_config();

	Interrupt_register(M_CAN_IRQ_NUM, M_CAN_interrupt_irq_handler);
	Interrupt_enable(M_CAN_IRQ_NUM);

	Interrupt_register(M_CAN_IRQ_NUM1, M_CAN_interrupt_irq_handler);
	Interrupt_enable(M_CAN_IRQ_NUM1);


	CAN_InitParams_t initParams = {0};
	CAN_FilterElement_t stdFiltelem = {0};

	initParams.fdMode = false;
	initParams.txbmode = CAN_TS_MODE_FIFO;
	initParams.timestamp = CAN_TIMESTAMP_EOF;

	return FDP_SUCCESS;
}

int M_CAN_Receiver(uint8_t *buf, uint32_t *index, uint32_t buf_len)
{
	MCAN_RxFIFOStatus_t fifoStatus = {0};
	MCAN_RxMessage_t RxMsgBuf = {0};
	uint32_t len = 0;

	MCAN_getNewDataStatus(M_CAN_BASE, &new_dat);

	if (new_dat.statusLow & (1U << 0)) {
		while(MCAN_receiveMsgFromBuffer(M_CAN_BASE, &RxMsgBuf, 0))
		{
			len = MCAN_getDataSize(RxMsgBuf.dlc);
			for (uint32_t i = 0; i < len; i++) {
			buf[(*index)++] = RxMsgBuf.data[i];
				if (*index >= buf_len)
					*index = 0;
			}
		}
	}

	if (new_dat.statusLow & (1U << 1)) {
		while(MCAN_receiveMsgFromBuffer(M_CAN_BASE, &RxMsgBuf, 1))
		{
			len = MCAN_getDataSize(RxMsgBuf.dlc);
			for (uint32_t i = 0; i < len; i++) {
				buf[(*index)++] = RxMsgBuf.data[i];
				if (*index >= buf_len)
					*index = 0;
			}
		}
	}

	if (new_dat.statusLow & (1U << 2)) {
		while(MCAN_receiveMsgFromBuffer(M_CAN_BASE, &RxMsgBuf, 2))
		{
			len = MCAN_getDataSize(RxMsgBuf.dlc);
			for (uint32_t i = 0; i < len; i++) {
				buf[(*index)++] = RxMsgBuf.data[i];
				if (*index >= buf_len)
					*index = 0;
			}
		}
	}

	return FDP_SUCCESS;
}

int M_CAN_Transmit(uint8_t *buf, uint32_t len)
{
	MCAN_TxMessage_t TxMsgBuf = {0};
	uint32_t i = 0;
	int datalen, maxlen;

	TxMsgBuf.brs = 0;
	TxMsgBuf.dlc = MCAN_DATA_LENGTH_8;
	TxMsgBuf.efc = 0;
	TxMsgBuf.esi = 0;
	TxMsgBuf.fdf = 0;
	TxMsgBuf.id = M_CAN_STANDARD_ID_W(0x456);
	TxMsgBuf.mm = 0x0;
	TxMsgBuf.mm1 = 0x0;
	TxMsgBuf.rtr = 0;
	TxMsgBuf.tsce = 0;
	TxMsgBuf.xtd = 0;

	if (len == 0)
		return FDP_SUCCESS;

	maxlen = MCAN_getDataSize(M_CAN_MSG_MAX_LEN);

	while (len) {

		if (len >= maxlen) {
			TxMsgBuf.dlc = M_CAN_MSG_MAX_LEN;
			len -= maxlen;
		} else {
			TxMsgBuf.dlc = len;
			len = 0;
		}

		datalen = MCAN_getDataSize(TxMsgBuf.dlc);
		memcpy(TxMsgBuf.data, &buf[i], datalen);
		i += datalen;

		while(!MCAN_transmitMsgBuffer(M_CAN_BASE, &TxMsgBuf, 0));

	}

	return FDP_SUCCESS;
}
