/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __FDP_PORT_H__
#define __FDP_PORT_H__

#include <stdbool.h>
#include <stdint.h>

/**
 * @brief User defined supported startup method device table.
 * 
 */
typedef enum {
	FDP_DEV_DEFAULT = 0,
	FDP_MEM_FLASH,
	FDP_DEV_UART,
	FDP_DEV_CAN_CTRL,
	FDP_DEV_I2C,
	FDP_DEV_SPI,
} fdp_device_type_t;

/**
 * @brief Initialize the communication devices.
 * 
 * @param type Communication device type.
 * @return int Return non-zero value means initialization fail, else success.
 */
int fdp_device_init(fdp_device_type_t type);

/**
 * @brief User defined supported device transmission API.
 * 
 * @param buf The device Sends frame to host data buffer.
 * @param len The device Sends frame to host data length.
 * @param type User defined supported startup method device table.
 * @return int return non-zero value means transmit fail, else success.
 */
int fdp_transmit(uint8_t *buf, uint32_t len, fdp_device_type_t type);

/**
 * @brief FDP receive data.
 * 
 * @param buf The buf parameter is a pointer to the data buffer.
 * @param index Record the read index of the receiving cache area.
 * @param type Communication device type.
 * @return int return non-zero value means receive fail, else success.
 */
int fdp_receive(uint8_t *buf, uint32_t *index, fdp_device_type_t type);

/**
 * @brief Obtain the maximum frame length supported by communication peripherals.
 * 
 * @param size Return the maximum frame length supported by communication peripherals.
 * @param type Communication device type.
 * @return int return non-zero value means get fail, else success.
 */
int fdp_get_fram_size(uint32_t *size, fdp_device_type_t type);

/**
 * @brief Disable the SOC/CPU global interrupt.
 * 
 */
void fdp_globale_interrupt_enable(void);

/**
 * @brief Enable the SOC/CPU global interrupt.
 * 
 */
void fdp_globale_interrupt_disable(void);

/**
 * @brief Preparing for obtaining boot mode.
 * 
 * @return int Return non-zero value means initialization fail, else success.
 */
int fdp_boot_sel_mode_init(void);

/**
 * @brief Get boot mode.
 * 
 * @param type Return obtain boot mode.
 * @return int Return non-zero value means get mode fail, else success.
 */
int fdp_boot_mode_get(fdp_device_type_t *type);

/**
 * @brief If it is not in upgrade boot mode,
 * jump directly to the set memory space to execute the code.
 * 
 * @param type Boot mode.
 */
void fdp_boot_jump_mem(fdp_device_type_t type);

#endif
