/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file    rampgen.h
 *   @brief   Ramp Generator Module
 *
 */

#ifndef RAMPGEN_H
#define RAMPGEN_H

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "inc/hw_types.h"
#include <math.h>
#include <stdint.h>

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/**
 * \brief          Defines the Ramp Signal Generator (RAMPGEN) structure
 *
 * \details        The rampgen can be used to generate a ramp signal
 *                 of a desired frequency
 */
typedef volatile struct {
    float32_t freq;         // Input: Ramp frequency (pu)
    float32_t stepAngleMax; // Parameter: Maximum step angle (pu)
    float32_t out;          // Output: Ramp signal (pu)
} RAMPGEN;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
/**
 * \brief      resets RAMPGEN internal storage data and outputs
 * \param v    a pointer to a RAMPGEN structure
 */
__STATIC_INLINE void
RAMPGEN_reset(RAMPGEN *v) {
    v->out = 0.0f;
}

/**
 * \brief      configures RAMPGEN module
 * \param v    a pointer to a RAMPGEN structure
 * \param isrFrequency Frequency at which the RAMGEN is called in ISR
 * \param rampFrequency Frequency of the RAMP generated by RAMPGEN module
 */
__STATIC_INLINE void
RAMPGEN_config(RAMPGEN *v,
               float32_t isrFrequency,
               float32_t rampFrequency) {
    v->freq = rampFrequency;
    v->stepAngleMax = ((float32_t)1.0) / isrFrequency;
}

/**
 * \brief      Run RAMPGEN module
 * \param v    a pointer to a RAMPGEN structure
 */
__STATIC_INLINE void
RAMPGEN_run(RAMPGEN *v) {
    //
    // Compute the angle rate
    //
    v->out += (v->stepAngleMax * v->freq);

    //
    // Saturate the angle rate within (0,1)
    //
    if (v->out > (1.0f)) {
        v->out -= (1.0f);
    }
}

#ifdef __cplusplus
}
#endif

#endif /*RAMPGEN_H*/
