#include <stdlib.h>
#include <string.h>

#include "device.h"

#include "driverlib.h"

#define EMBEDDED_CLI_IMPL
#include "embedded_cli.h"

#include "printf.h"

__attribute__ ((section (".RamFunc"))) static void Reboot(EmbeddedCli *cli, char *args, void *context)
{
    volatile int delay = 1000*1000;

    __disable_irq();

    SysCtl_setSrstEflashCtrlN(0);
    SysCtl_setSrstEflashCtrlN(1);

    while(delay --);

    *((volatile uint32_t *)(0x40007020)) = 0;   //switch to main flash

    delay = 1000;
    while(delay --);

    SysCtl_setSrstN(1);
    SysCtl_setSrstN(0);
}

static void Pinmux(EmbeddedCli *cli, char *args, void *context)
{
    uint32_t needHelp = 1;
    uint16_t val, pin;

    if (embeddedCliGetTokenCount(args) < 2) {
        goto TestPinmux_Help;
    }

    pin = atoi(embeddedCliGetToken(args, 1));

    if (pin > 62) {
        printf("pin number outof range [0, 62], enter \"gpio\" for help.\r\n");
        return;
    }

    if (strcmp("get", embeddedCliGetToken(args, 2)) == 0) {
        val = SysCtl_PinMux_getIOMux(pin);
        printf("get pinmux %d func_id is %d\r\n", pin, val);
    } else if (strcmp("set", embeddedCliGetToken(args, 2)) == 0) {
        if (embeddedCliGetTokenCount(args) < 3) {
            goto TestPinmux_Help;
        }

        uint32_t func_id = atoi(embeddedCliGetToken(args, 3));

        if (func_id > 15) {
            printf("func_id %d outof range [0, 15].\r\n", val);
            return;
        }

        SysCtl_PinMux_setIOMux(pin, func_id);
    } else {
        printf("invalid parameter: %s\r\n", embeddedCliGetToken(args, 2));
    }

    return;

TestPinmux_Help:
    printf("read write pinmux setting.\r\n");
    printf("usage: pinmux [pin_number] [get/set] [func_id].\r\n");
    printf("example: pinmux 0 get\r\n");
    printf("example: pinmux 0 set 1\r\n");
}

static void AnaSel(EmbeddedCli *cli, char *args, void *context)
{
    uint32_t needHelp = 1;
    uint16_t val, pin;

    if (embeddedCliGetTokenCount(args) < 2) {
        goto AnaSel_Help;
    }

    char name = embeddedCliGetToken(args, 1)[0];
    uint32_t ch = atoi(embeddedCliGetToken(args, 1)+1);
    uint32_t regBase;

    if( name == 'a' || name == 'A') {
        if (ch > 15) {
            printf("unknown analog pin: %c%d", name, ch);
            return;
        }

        regBase = 0x40350000 + 0x0204 + 4*ch;
    } else if (name == 'b' || name == 'B') {
        switch (ch) {
            case 0: regBase = 0x40350000 + 0x0238; break;
            case 2: regBase = 0x40350000 + 0x023C; break;
            case 3: regBase = 0x40350000 + 0x0240; break;
            case 4: regBase = 0x40350000 + 0x0244; break;
            case 5: regBase = 0x40350000 + 0x0248; break;
            case 11: regBase = 0x40350000 + 0x024C; break;
            case 12: regBase = 0x40350000 + 0x0250; break;
            default:
                if (ch > 15) {
                    printf("unknown analog pin: %c%d.\r\n", name, ch);
                } else {
                    printf("analog pin %c%d doesn't has function selection register.\r\n", name, ch);
                }
                return;
        }
    } else if (name == 'c' || name == 'C') {
        if (ch == 1) {
            regBase = 0x40350000 + 0x0254;
        } else if(ch == 3) {
            regBase = 0x40350000 + 0x0258;
        } else if(ch == 14) {
            regBase = 0x40350000 + 0x025C;
        } else {
            if (ch > 15) {
                printf("unknown analog pin: %c%d.\r\n", name, ch);
            } else {
                printf("analog pin %c%d doesn't has function selection register.\r\n", name, ch);
            }
            return;
        }

    } else {
        printf("invalid analog pin number: %s\r\n", embeddedCliGetToken(args, 1));
        return;
    }

    if (strcmp("get", embeddedCliGetToken(args, 2)) == 0) {
        val = HWREG(regBase);
        printf("get analog pin %c%d function selected: %d\r\n", name, ch, val&0x0F);
    } else if (strcmp("set", embeddedCliGetToken(args, 2)) == 0) {
        if (embeddedCliGetTokenCount(args) < 3) {
            goto AnaSel_Help;
        }

        uint32_t func_id = atoi(embeddedCliGetToken(args, 3));

        if (func_id > 15) {
            printf("func_id %d outof range [0, 15].\r\n", val);
            return;
        }

        HWREG(regBase) = func_id;
        printf("set analog pin %c%d function id: %d ok.\r\n", name, ch, func_id);
    } else {
        printf("invalid parameter: %s\r\n", embeddedCliGetToken(args, 2));
    }

    return;

AnaSel_Help:
    printf("read write analog pin function selection setting.\r\n");
    printf("usage: ana_sel [pin_number] [get/set] [func_id].\r\n");
    printf("param [pin_number]: A0~A15, B0, B2, B3, B4, B5, B11, B12, C1, C3, C14\r\n");
    printf("example: ana_sel a0 get\r\n");
    printf("example: ana_sel a0 set 0\r\n");
}

static void GpioReadWrite(EmbeddedCli *cli, char *args, void *context)
{
    uint32_t needHelp = 1;
    uint16_t val, pin;

    if (embeddedCliGetTokenCount(args) < 2) {
        goto GpioReadWrite_Help;
    }

    pin = atoi(embeddedCliGetToken(args, 1));

#ifdef GS32F00xx
    if (pin > 62) {
        printf("pin number outof range [0, 62], enter \"gpio\" for help.\r\n");
        return;
    }
#elif defined(GS32F3xx)
    if (pin > 99 && pin != 133) {
        printf("pin number outof range [0..99, 133], enter \"gpio\" for help.\r\n");
        return;
    }
#endif

    if (strcmp("in", embeddedCliGetToken(args, 2)) == 0) {
        GPIO_disableWritePin(pin);
        val = GPIO_getPinData(pin);

        printf("get gpio%d input: %d\r\n", pin, val);
        return;
    } else {
        if (embeddedCliGetTokenCount(args) < 3) {
            goto GpioReadWrite_Help;
        }
    }

    //now we have 3 args
    if (strcmp("out", embeddedCliGetToken(args, 2)) == 0) {
        val = atoi(embeddedCliGetToken(args, 3));
        GPIO_enableWritePin(pin);

        if (val) {
            GPIO_writePin(pin, 1);
        } else {
            GPIO_writePin(pin, 0);
        }

        printf("set gpio%d output to %d\r\n", pin, val?1:0);

    } else if (strcmp("pull", embeddedCliGetToken(args, 2)) == 0) {
        if (strcmp("up", embeddedCliGetToken(args, 3)) == 0) {
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
            GPIO_setPadConfig(pin, GPIO_PIN_TYPE_PULLUP);
#elif IS_GS32F00xx(0x11)
            SysCtl_PinMux_setPE(pin, 1);
            SysCtl_PinMux_setPS(pin, IO_SET_PS_TYPE_PULL_UP);
#endif
        } else if (strcmp("down", embeddedCliGetToken(args, 3)) == 0) {
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
            GPIO_setPadConfig(pin, GPIO_PIN_TYPE_PULLDOWN);
#elif IS_GS32F00xx(0x11)
            SysCtl_PinMux_setPE(pin, 1);
            SysCtl_PinMux_setPS(pin, IO_SET_PS_TYPE_PULL_DOWN);
#endif
        } else if (strcmp("none", embeddedCliGetToken(args, 3)) == 0) {
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
            GPIO_setPadConfig(pin, GPIO_PIN_TYPE_STD);
#elif IS_GS32F00xx(0x11)
            SysCtl_PinMux_setPE(pin, 0);
#endif
        } else {
            printf("invalid parameter: %s, should be up/down/none\r\n", embeddedCliGetToken(args, 3));
            return;
        }
    } else if (strcmp("drv", embeddedCliGetToken(args, 2)) == 0) {
        val = atoi(embeddedCliGetToken(args, 3));

        if (val > 3) {
            printf("drv level %d outof range [0,3].\r\n", val);
            return;
        }

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
        GPIO_setStrength(pin, val);
#else
        SysCtl_PinMux_setDS(pin, val);
#endif
    } else {
        printf("invalid command param: %s, enter \"gpio\" for help.\r\n", embeddedCliGetToken(args, 2));
        return;
    }

    return;

GpioReadWrite_Help:
    printf("read/write gpio\r\n");
    printf("usage: gpio [pin_number: 0~62] [in/out/pull/drv] [value]\r\n");
    printf("example: gpio 0 in\r\n");
    printf("example: gpio 2 out 1\r\n");
    printf("example: gpio 5 pull up\r\n");
    printf("example: gpio 7 pull down\r\n");
    printf("example: gpio 7 pull none\r\n");
    printf("example: gpio 22 drv 2\r\n");
}

static float GetAnalogVoltage(uint32_t adcBase, uint32_t resultBase, uint32_t channel, uint16_t *adcResult)
{
    ADC_setPrescaler(adcBase, ADC_CLK_DIV_3_0);
    ADC_setInterruptPulseMode(adcBase, ADC_PULSE_END_OF_CONV);
    ADC_enableConverter(adcBase);

    volatile uint32_t delay = 0xffff;
    while(delay --);

    ADC_disableBurstMode(adcBase);
    ADC_setSOCPriority(adcBase, ADC_PRI_ALL_ROUND_ROBIN);

    ADC_setupSOC(adcBase, ADC_SOC_NUMBER0, ADC_TRIGGER_SW_ONLY, channel, 50U);
    ADC_setInterruptSOCTrigger(adcBase, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);

    ADC_setInterruptSource(adcBase, ADC_INT_NUMBER1, ADC_SOC_NUMBER0);
    ADC_enableInterrupt(adcBase, ADC_INT_NUMBER1);
    ADC_clearInterruptStatus(adcBase, ADC_INT_NUMBER1);
    ADC_disableContinuousMode(adcBase, ADC_INT_NUMBER1);

    ADC_forceMultipleSOC(adcBase, ADC_FORCE_SOC0);

    while(!ADC_getInterruptStatus(adcBase, ADC_INT_NUMBER1));   //wait adc convert
    ADC_clearInterruptStatus(adcBase, ADC_INT_NUMBER1);

    uint16_t result = ADC_readResult(resultBase, ADC_SOC_NUMBER0);
    float volt = 3.3f * (float)result/4096.0;

    *adcResult = result;
    return volt;
}

static void ReadAdc(EmbeddedCli *cli, char *args, void *context)
{
    uint32_t needHelp = 1;
    uint16_t val, ch;
    uint32_t adcBase, resultBase;

    if (embeddedCliGetTokenCount(args) < 1) {
        goto TestGpio_Help;
    }

    char name = embeddedCliGetToken(args, 1)[0];
    ch = atoi(embeddedCliGetToken(args, 1)+1);

    if (ch > 15) {
        printf("analog pin %s outof range [a0/b0/c0, a15/b15/c15]\r\n", embeddedCliGetToken(args, 1));
        return;
    }

    if( name == 'a' || name == 'A') {
        adcBase = ADCA_BASE;
        resultBase = ADCARESULT_BASE;
    } else if (name == 'b' || name == 'B') {
        adcBase = ADCB_BASE;
        resultBase = ADCBRESULT_BASE;
    } else if (name == 'c' || name == 'C') {
        adcBase = ADCC_BASE;
        resultBase = ADCCRESULT_BASE;
    }
#if defined(ADCD_BASE)
    else if (name == 'd' || name == 'D') {
        adcBase = ADCD_BASE;
        resultBase = ADCDRESULT_BASE;
    }
#endif
    else {
        printf("invalid analog pin number: %s\r\n", embeddedCliGetToken(args, 1));
        return;
    }

#if IS_GS32F3xx(0x20)
    //config for gj320f3xx
    //
    ADC_setVREF(ADCA_BASE,ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);
    ADC_setVREF(ADCB_BASE,ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);
    ADC_setVREF(ADCC_BASE,ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);
    ADC_setVREF(ADCD_BASE,ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);
#else

#if IS_GS32F00xx(0x11)
    HWREG(ADCA_BASE + 0xe0) = 0x888;
    HWREG(ADCB_BASE + 0xe0) = 0x888;
    HWREG(ADCC_BASE + 0xe0) = 0x888;
    ASysCtl_setAdcTopSpare(0x5F);
    ASysCtl_setAnaClkSpare(0xA1);

    ADC_setVCM(ADCA_BASE,ADC_VCM_1_65V);
    ADC_setVCM(ADCB_BASE,ADC_VCM_1_65V);
    ADC_setVCM(ADCC_BASE,ADC_VCM_1_65V);
#endif
    ADC_setVREF(ADCA_BASE,ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);
    ADC_setVREF(ADCB_BASE,ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);
    ADC_setVREF(ADCC_BASE,ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);
#endif

#if 0
    if(strcmp("all", embeddedCliGetToken(args, 1)) == 0) {
#ifdef GS32F00xx
        for (uint32_t ch=0; ch<16; ch+=1) {
#else
		for (uint32_t ch=0; ch<8; ch+=1) {
#endif
        	uint16_t result;
            float volt = GetAnalogVoltage(ADCA_BASE, ADCARESULT_BASE, ch, &result);
            printf("A%02d: %d / %.3f V", ch, result, volt);
            volt = GetAnalogVoltage(ADCA_BASE, ADCARESULT_BASE, ch, &result);
            printf("A%02d: %d / %.3f V", ch, result, volt);

            volt = GetAnalogVoltage(ADCB_BASE, ADCBRESULT_BASE, ch, &result);
            printf("B%02d: %d / %.3f V", ch, result, volt);
            volt = GetAnalogVoltage(ADCB_BASE, ADCBRESULT_BASE, ch, &result);
            printf("B%02d: %d / %.3f V", ch, result, volt);

            volt = GetAnalogVoltage(ADCC_BASE, ADCCRESULT_BASE, ch, &result);
            printf("C%02d: %d / %.3f V", ch, result, volt);
            volt = GetAnalogVoltage(ADCC_BASE, ADCCRESULT_BASE, ch, &result);
            printf("C%02d: %d / %.3f V", ch, result, volt);

            #if defined(ADCD_BASE)
            volt = GetAnalogVoltage(ADCD_BASE, ADCDRESULT_BASE, ch, &result);
            printf("D%02d: %d / %.3f V", ch, result, volt);
            volt = GetAnalogVoltage(ADCD_BASE, ADCDRESULT_BASE, ch, &result);
            printf("D%02d: %d / %.3f V", ch, result, volt);
            #endif
            printf("\r\n");

        }
    }
#else
        if(strcmp("all", embeddedCliGetToken(args, 1)) == 0) {

            	uint16_t result;
                float volt = 0;
                volt= GetAnalogVoltage(ADCB_BASE, ADCARESULT_BASE, 3, &result);
                printf("B3:%.3fV======================",volt);
                volt = GetAnalogVoltage(ADCA_BASE, ADCBRESULT_BASE, 6, &result);
                printf("A6:%.3fV\r\n\r\n",volt);

                volt = GetAnalogVoltage(ADCA_BASE, ADCARESULT_BASE, 2, &result);
                printf("A2:%.3fV======================",volt);
                volt = GetAnalogVoltage(ADCB_BASE, ADCBRESULT_BASE, 2, &result);
                printf("B2:%.3fV\r\n\r\n",volt);


                volt = GetAnalogVoltage(ADCA_BASE, ADCARESULT_BASE, 15, &result);
                printf("A15:%.3fV======================",volt);
                volt = GetAnalogVoltage(ADCB_BASE, ADCBRESULT_BASE, 3, &result);
                printf("B3:%.3fV\r\n\r\n",volt);


                volt = GetAnalogVoltage(ADCB_BASE, ADCARESULT_BASE, 14, &result);
                printf("B14:%.3fV======================",volt);
                volt = GetAnalogVoltage(ADCB_BASE, ADCBRESULT_BASE, 9, &result);
                printf("B9:%.3fV\r\n\r\n",volt);


                volt = GetAnalogVoltage(ADCA_BASE, ADCARESULT_BASE, 11, &result);
                printf("A11:%.3fV======================",volt);
                volt = GetAnalogVoltage(ADCB_BASE, ADCBRESULT_BASE, 12, &result);
                printf("B12:%.3fV\r\n\r\n",volt);


                volt = GetAnalogVoltage(ADCC_BASE, ADCARESULT_BASE, 2, &result);
                printf("C2:%.3fV======================",volt);
                volt = GetAnalogVoltage(ADCC_BASE, ADCBRESULT_BASE, 1, &result);
                printf("C1:%.3fV\r\n\r\n",volt);


                volt = GetAnalogVoltage(ADCA_BASE, ADCARESULT_BASE, 1, &result);
                printf("A1:%.3fV======================",volt);
                volt = GetAnalogVoltage(ADCA_BASE, ADCBRESULT_BASE, 7, &result);
                printf("A7:%.3fV\r\n\r\n",volt);


                volt = GetAnalogVoltage(ADCA_BASE, ADCARESULT_BASE, 0, &result);
                printf("A0:%.3fV======================",volt);
                volt = GetAnalogVoltage(ADCA_BASE, ADCBRESULT_BASE, 12, &result);
                printf("A12:%.3fV\r\n\r\n",volt);

                volt = GetAnalogVoltage(ADCA_BASE, ADCBRESULT_BASE, 8, &result);
                printf("A8:%.3fV======================",volt);
                volt = GetAnalogVoltage(ADCB_BASE, ADCBRESULT_BASE, 4, &result);
                printf("B4:%.3fV\r\n\r\n",volt);

                volt = GetAnalogVoltage(ADCB_BASE, ADCARESULT_BASE, 5, &result);
                printf("B5:%.3fV======================",volt);
                volt = GetAnalogVoltage(ADCA_BASE, ADCBRESULT_BASE, 10, &result);
                printf("A10:%.3fV\r\n\r\n",volt);

                volt = GetAnalogVoltage(ADCA_BASE, ADCARESULT_BASE, 5, &result);
                printf("A5:%.3fV\r\n\r\n",volt);

                volt = GetAnalogVoltage(ADCA_BASE, ADCARESULT_BASE, 4, &result);
                printf("A4:%.3fV\r\n\r\n",volt);

                volt = GetAnalogVoltage(ADCA_BASE, ADCBRESULT_BASE, 9, &result);
                printf("A9:%.3fV\r\n\r\n",volt);


                volt = GetAnalogVoltage(ADCB_BASE, ADCARESULT_BASE, 0, &result);
                printf("B0:%.3fV\r\n\r\n",volt);
                volt = GetAnalogVoltage(ADCC_BASE, ADCBRESULT_BASE, 14, &result);
                printf("C14:%.3fV\r\n\r\n",volt);


                #if defined(ADCD_BASE)
                volt = GetAnalogVoltage(ADCD_BASE, ADCDRESULT_BASE, ch, &result);
                printf("D%02d: %d / %.3f V", ch, result, volt);
                volt = GetAnalogVoltage(ADCD_BASE, ADCDRESULT_BASE, ch, &result);
                printf("D%02d: %d / %.3f V", ch, result, volt);
                #endif
                printf("\r\n");

        }
#endif
         else if (embeddedCliGetToken(args, 1)[1] != '\0') {
    	uint16_t result;
        float volt = GetAnalogVoltage(adcBase, resultBase, ch, &result);
        volt = GetAnalogVoltage(adcBase, resultBase, ch, &result);
        printf("%c%d: %d / %.3f V\r\n", name, ch, result, volt);
    } else {
#ifdef GS32F00xx
        for (uint32_t ch=0; ch<16; ch+=1) {
#else
		for (uint32_t ch=0; ch<8; ch+=1) {
#endif
        	uint16_t result;
            float volt = GetAnalogVoltage(adcBase, resultBase, ch, &result);
            volt = GetAnalogVoltage(adcBase, resultBase, ch, &result);
            printf("%c%02d: %d / %.3f V\r\n", name, ch, result, volt);
        }
    }

    return;

TestGpio_Help:
    printf("read analog pin input voltage\r\n");
    printf("usage: adc [channel number: a0~c16, a/b/c/all]\r\n");
    printf("example: adc a2\r\n");
    printf("example: adc a\r\n");
    printf("example: adc all\r\n");
}

static void ShowData(void *buf, uint32_t baseAddr, uint32_t width, uint32_t items)
{
    for (uint32_t i=0; i<items; i+=1) {
        if (i%8==0) {
            printf("%08X: ", baseAddr + i*width);
        }

        switch(width) {
            case 1: printf("%02X ", ((uint8_t *)buf)[i]); break;
            case 2: printf("%04X ", ((uint16_t *)buf)[i]); break;
            case 4: printf("%08X ", ((uint32_t *)buf)[i]); break;
        }

        if (i%8==7) {
            printf("\r\n");
        }
    }
}

static void ReadMem(EmbeddedCli *cli, char *args, void *context)
{
    uint8_t infoBuf[64];

    if (embeddedCliGetTokenCount(args) < 3) {
        goto ReadInfo_Help;
    }

    uint8_t width = atoi(embeddedCliGetToken(args, 1));

    if (width != 8 && width != 16 && width != 32) {
        printf("parameter \"%d\" error, should be 8/16/32\r\n", width);
        return;
    }

    uint32_t addr = strtol(embeddedCliGetToken(args, 2), NULL, 16);

    uint32_t items = atoi(embeddedCliGetToken(args, 3));
    width = width/8;

    uint32_t len = 0;

    while (len < items * width) {
        uint32_t readLen = items * width - len;

        if (readLen > sizeof(infoBuf)) {
            readLen = sizeof(infoBuf);
        }

        if (width == 32) {
            for (uint32_t i=0; i<readLen/4; i+=1) {
                ((uint32_t *)infoBuf)[i] = ((uint32_t *)addr)[i];
            }
        } else if (width == 16) {
            for (uint32_t i=0; i<readLen/2; i+=1) {
                ((uint16_t *)infoBuf)[i] = ((uint16_t *)addr)[i];
            }
        } else {
            memcpy(infoBuf, (void *)addr, readLen);
        }

        ShowData(infoBuf, addr, width, readLen/width);

        addr += readLen;
        len += readLen;
    }

    if (items % 8 != 0) {
        printf("\r\n");
    }

    return;

    ReadInfo_Help:
    printf("read mem content.\r\n");
    printf("usage: mem [8/16/32] [Addr_hex] [items_dec]\r\n");
    printf("example: mem 8 20000000 16\r\n");
    printf("example: mem 32 20000000 16\r\n");
    printf("example: mem 32 40370048 1\r\n");
}


static void OBSClk(EmbeddedCli *cli, char *args, void *context)
{
    if (embeddedCliGetTokenCount(args) < 2) {
        goto OBSClk_Help;
    }

    uint32_t id = atoi(embeddedCliGetToken(args, 1));
    uint32_t div = atoi(embeddedCliGetToken(args, 2));

    if (id > 15) {
        printf("source_id out of range [0, 15]\r\n");
        return;
    }


    SysCtl_setObsSigSel(id);

    div /= 2;

    if (div > 0x7F) {
        div = 0x7F;
    }
    SysCtl_setObsSigDiv(div);

#if IS_GS32F00xx(0x11)
    SysCtl_PinMux_setDS(16, IO_SET_DS_TYPE_3);
#elif IS_GS32F00xx(0x12)
    GPIO_setPinConfig(GPIO_16_XCLKOUT);
    GPIO_setStrength(16, GPIO_PIN_DRIVE_STRENGTH_3);
#elif IS_GS32F3xx(0x22)
    GPIO_setPinConfig(GPIO_73_XCLKOUT);
    GPIO_setStrength(73, GPIO_PIN_DRIVE_STRENGTH_3);
#endif

    return;

OBSClk_Help:
    printf("setup obs clock.\r\n");
    printf("usage: obsclk [source_id] [div]\r\n");
    printf("example: obsclk 2 36\r\n");
    printf("source_id:\r\n");

#if IS_GS32F00xx(0x12)
    printf("SYS_CLK_PRE_DIV = 0\r\n");
    printf("      FLASH_SMW = 1\r\n");
    printf("       CAN_CCLK = 2\r\n");
    printf("        APB_CLK = 3\r\n");
    printf("        DSP_CLK = 4\r\n");
    printf("        ARM_CLK = 5\r\n");
    printf("   PLL_OUT_DIV2 = 8\r\n");
    printf("        PLL_OUT = 9\r\n");
    printf("        REF_PIN = 10\r\n");
    printf("           OSC1 = 11\r\n");
    printf("           OSC2 = 12\r\n");
    printf("        PLL_REF = 13\r\n");
#elif IS_GS32F00xx(0x11)
    printf("PLL_MUX_OUT = 0\r\n");
    printf("PLL_OUT_CLK = 1\r\n");
    printf("BPLL_OUT_CLK = 2\r\n");
    printf("BPLL_FOUTVCO = 3\r\n");
    printf("BPLL_FOUTPOSTDIV = 4\r\n");
    printf("CLKOUT_ANALOG_CLK = 5\r\n");
    printf("CLKOUT_CAN_CLK = 6\r\n");
    printf("AHB_PERI_CLK = 7\r\n");
    printf("FREE_CLK = 8\r\n");
    printf("ALL_ON_L_CLK = 9\r\n");
    printf("HSI_OBS_CLK = 10\r\n");
    printf("ALL_ON_H_CLK = 11\r\n");
    printf("AF_BUF_LSE_CLK = 12\r\n");
#elif IS_GS32F3xx(0x22)
    printf("        PLL_OUT = 0\r\n");
    printf("    PLL_PostDiv = 1\r\n");
    printf("        PLL_VCO = 2\r\n");
    printf("    PLL_Ref_CLK = 3\r\n");
    printf("           OSC2 = 4\r\n");
    printf("           OSC1 = 5\r\n");
    printf("        PIN_CLK = 6\r\n");
    printf("   CAN_CORE_CLK = 7\r\n");
#endif
}

static void CANSwitch(EmbeddedCli *cli, char *args, void *context)
{
    if (embeddedCliGetTokenCount(args) < 1) {
        goto CanSw_Help;
    }

    uint32_t id = atoi(embeddedCliGetToken(args, 1));
    SysCtl_setCanClkSrcSel(id);

    return;
CanSw_Help:
	printf("CAN_CLK_TYPE_HSE 		= 	0\r\n");
	printf("CAN_CLK_TYPE_AUXCLK_IN	=	1\r\n");
	printf("CAN_CLK_TYPE_PLL 		= 	2\r\n");

}



REGISTER_CLI_COMMAND( "obsclk", "setup obs clock output", true, NULL, OBSClk);
REGISTER_CLI_COMMAND( "CANSwitch", "switch can clock sorce", true, NULL, CANSwitch);

REGISTER_CLI_COMMAND( "reboot", "reboot", false, NULL, Reboot);
REGISTER_CLI_COMMAND( "pinmux", "read write pinmux settings", true, NULL, Pinmux);
REGISTER_CLI_COMMAND( "ana_sel", "read write analog pin settings", true, NULL, AnaSel);
REGISTER_CLI_COMMAND( "gpio", "read write gpio", true, NULL, GpioReadWrite);
REGISTER_CLI_COMMAND( "adc", "read analog pin input voltage", true, NULL, ReadAdc);
REGISTER_CLI_COMMAND( "mem", "read memory", true, NULL, ReadMem);

