/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    encoder.h
*   @brief   encoder header file
*/


#ifndef DEVICE_DRIVERLIB_ENCODER_H_
#define DEVICE_DRIVERLIB_ENCODER_H_

#ifdef __cplusplus
extern "C"{
#endif

#if 1

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "inc/hw_types.h"
#include "inc/hw_encoder.h"
#include "inc/hw_memmap.h"
#include "debug.h"
#include "board_cfg.h"


/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
typedef enum
{
	TFORMAT = 0U,
	AFORMAT = 1U
}TA_FORMAT;

typedef enum
{
	HARDWARE = 0U,
	SOFTWARE = 1U
}RCV_MODE;

typedef enum
{
	INVALID = 0U,
	SPIA = 1U,
	SPIB = 2U
}RXD_INPUTSIGNAL;

typedef enum
{
	BISS = 0U,
	SSI = 1U
}BISS_SSI_MODE;

typedef enum
{
	NONE = 0U,
	SPIA_SIMO = 1U,
	SPIB_SIMO = 2U,
	CLB_INPUT_XBAR1 = 8U,
	CLB_INPUT_XBAR2 = 9U,
	CLB_INPUT_XBAR3 = 10U,
	CLB_INPUT_XBAR4 = 11U,
	CLB_INPUT_XBAR5 = 12U,
	CLB_INPUT_XBAR6 = 13U,
	CLB_INPUT_XBAR7 = 14U,
	CLB_INPUT_XBAR8 = 15U,
}QEPDIV_INPUT_SEL;
/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
#if IS_GS32F00xx(0x30)
#ifdef DEBUG
__STATIC_INLINE bool
ENC_isBaseValid(uint32_t base)
{
    return(
           (base == ENC_BASE)
          );
}
#endif

//***********************************     AF_FORMAT_START     ***********************************//

__STATIC_INLINE void ENC_TA_setTxNum(uint32_t base,uint16_t num)
{
	ASSERT(ENC_isBaseValid(base));
	ASSERT((num<=0xF));
	
	HWREG(base + TA_FORMAT_CTRL1) =  (HWREG(base + TA_FORMAT_CTRL1) & \
			~ENC_APB_CFG_TA_TX_FRM_NUM_M) | (num<<ENC_APB_CFG_TA_TX_FRM_NUM_S);
}

__STATIC_INLINE void ENC_TA_setRxNum(uint32_t base,uint32_t num)
{
	ASSERT(ENC_isBaseValid(base));
	ASSERT((num<=0xF));
	
	HWREG(base + TA_FORMAT_CTRL1) =  (HWREG(base + TA_FORMAT_CTRL1) & \
			~ENC_APB_CFG_TA_RX_FRM_NUM_M) | (num<<ENC_APB_CFG_TA_RX_FRM_NUM_S);
}

__STATIC_INLINE void ENC_TA_setRxLen(uint32_t base,uint32_t num)
{
	ASSERT(ENC_isBaseValid(base));
	ASSERT((num<=0xFF));

	HWREG(base + TA_FORMAT_CTRL1) =  (HWREG(base + TA_FORMAT_CTRL1) & \
			~ENC_APB_CFG_TA_RX_FRM_LEN_M) | (num<<ENC_APB_CFG_TA_RX_FRM_LEN_S);
}

__STATIC_INLINE void ENC_TA_setTxLen(uint32_t base,uint32_t num)
{
	ASSERT(ENC_isBaseValid(base));
	ASSERT((num<=0xFF));

	HWREG(base + TA_FORMAT_CTRL1) =  (HWREG(base + TA_FORMAT_CTRL1) & \
			~ENC_APB_CFG_TA_TX_FRM_LEN_M) | (num<<ENC_APB_CFG_TA_TX_FRM_LEN_S);
}

__STATIC_INLINE void ENC_TA_setBaudrate(uint32_t base,uint16 baudrate)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_CTRL0) =  (HWREG(base + TA_FORMAT_CTRL0) & \
			~ENC_APB_CFG_TA_BAUDRATE_M) | (baudrate<<ENC_APB_CFG_TA_BAUDRATE_S);
}

__STATIC_INLINE void ENC_TA_selRxdIn(uint32_t base,RXD_INPUTSIGNAL signal)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_CTRL0) =  (HWREG(base + TA_FORMAT_CTRL0) & \
			~ENC_APB_CFG_TA_RXD_IN_SEL_M) | (signal << ENC_APB_CFG_TA_RXD_IN_SEL_S);
}

__STATIC_INLINE void ENC_TA_selRcvMode(uint32_t base,RCV_MODE mode)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_CTRL0) =  (HWREG(base + TA_FORMAT_CTRL0) & \
			~ENC_APB_CFG_TA_RCV_MODE_M) | (mode<<ENC_APB_CFG_TA_RCV_MODE_S);
}

__STATIC_INLINE void ENC_TA_selFormat(uint32_t base,TA_FORMAT format)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_CTRL0) =  (HWREG(base + TA_FORMAT_CTRL0) & \
			~ENC_APB_CFG_TA_FORMAT_M) | (format<<ENC_APB_CFG_TA_FORMAT_S);
}

__STATIC_INLINE void ENC_TA_Enable(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_CTRL0) |=  ENC_APB_CFG_TA_EN_M;
}

__STATIC_INLINE void ENC_TA_Disable(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_CTRL0) &=  (~ENC_APB_CFG_TA_EN_M);
}


__STATIC_INLINE uint16_t ENC_TA_getTimeofWaitRpt(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	return ((HWREG(base + TA_FORMAT_STATUS) & \
			ENC_APB_CFG_TA_WAIT_TIME_RPT_M)>>ENC_APB_CFG_TA_WAIT_TIME_RPT_S);
}

__STATIC_INLINE uint16_t ENC_TA_getAFState(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	return (HWREG(base + TA_FORMAT_STATUS) & ENC_APB_CFG_TA_STATE_M);
}

__STATIC_INLINE void ENC_TA_setTime2(uint32_t base,uint32_t t2)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_TIMING0) =  (HWREG(base + TA_FORMAT_TIMING0) & \
			~ENC_APB_CFG_TA_T2_CFG_M) | (t2 << ENC_APB_CFG_TA_T2_CFG_S);
}

__STATIC_INLINE void ENC_TA_setTime3(uint32_t base,uint32_t t3)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_TIMING0) =  (HWREG(base + TA_FORMAT_TIMING0) & \
			~ENC_APB_CFG_TA_T3_CFG_M) | (t3 << ENC_APB_CFG_TA_T3_CFG_S);
}

__STATIC_INLINE void ENC_TA_setTime4(uint32_t base,uint32_t t4)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_TIMING1) =  (HWREG(base + TA_FORMAT_TIMING1) & \
			~ENC_APB_CFG_TA_T4_CFG_M) | (t4 << ENC_APB_CFG_TA_T4_CFG_S);
}

__STATIC_INLINE void ENC_TA_setTime5(uint32_t base,uint32_t t5)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_TIMING1) =  (HWREG(base + TA_FORMAT_TIMING1) & \
			~ENC_APB_CFG_TA_T5_CFG_M) | (t5 << ENC_APB_CFG_TA_T5_CFG_S);
}

__STATIC_INLINE void ENC_TA_setTime6(uint32_t base,uint32_t t6)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_TIMING2) =  (HWREG(base + TA_FORMAT_TIMING2) & \
			~ENC_APB_CFG_TA_T6_CFG_M) | (t6 << ENC_APB_CFG_TA_T6_CFG_S);
}

__STATIC_INLINE void ENC_TA_setTime9(uint32_t base,uint32_t t9)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + TA_FORMAT_TIMING2) =  (HWREG(base + TA_FORMAT_TIMING2) & \
			~ENC_APB_CFG_TA_T9_CFG_M) | (t9 << ENC_APB_CFG_TA_T9_CFG_S);
}
//***********************************     AF_FORMAT_END     ***********************************//



//***********************************     BISS_START     ***********************************//

__STATIC_INLINE void ENC_BISS_Enable(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_CTRL) |=	ENC_APB_CFG_BISS_EN_M;
}

__STATIC_INLINE void ENC_BISS_Disable(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_CTRL) &=	(~ENC_APB_CFG_BISS_EN_M);
}

__STATIC_INLINE void ENC_BISS_SPI_MOD(uint32_t base,BISS_SSI_MODE mode)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_CTRL) =	(HWREG(base + BISS_CTRL) & \
			(~ENC_APB_CFG_BISS_SSI_MODE_M)) | (mode << ENC_APB_CFG_BISS_SSI_MODE_S);
}

__STATIC_INLINE void ENC_BISS_selRcvMode(uint32_t base,RCV_MODE mode)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_CTRL) =  (HWREG(base + BISS_CTRL) & \
			(~ENC_APB_CFG_BISS_RCV_MODE_M)) | (mode<<ENC_APB_CFG_BISS_RCV_MODE_S);
}

__STATIC_INLINE void ENC_BISS_setCDM(uint32_t base,uint32_t cdm)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_CTRL) =  (HWREG(base + BISS_CTRL) & \
			(~ENC_APB_CFG_BISS_CDM_BIT_M)) | (cdm<<ENC_APB_CFG_BISS_CDM_BIT_S);
}

__STATIC_INLINE void ENC_BISS_selRxdIn(uint32_t base,RXD_INPUTSIGNAL signal)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_CTRL) =  (HWREG(base + BISS_CTRL) & \
			(~ENC_APB_CFG_BISS_SL_IN_SEL_M)) | (signal << ENC_APB_CFG_BISS_SL_IN_SEL_S);
}

__STATIC_INLINE void ENC_BISS_setBaudrate(uint32_t base,uint32_t baudrate)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_CTRL) =  (HWREG(base + BISS_CTRL) & \
			(~ENC_APB_CFG_BISS_BAUDRATE_M)) | (baudrate<<ENC_APB_CFG_BISS_BAUDRATE_S);
}

__STATIC_INLINE void ENC_BISS_setRxNum(uint32_t base,uint32_t num)
{
	ASSERT(ENC_isBaseValid(base));
	ASSERT((num<=0xFF));
	
	HWREG(base + BISS_CTRL) =  (HWREG(base + BISS_CTRL) & \
			(~ENC_APB_CFG_BISS_RX_BIT_NUM_M)) | (num<<ENC_APB_CFG_BISS_RX_BIT_NUM_S);
}

__STATIC_INLINE void ENC_BISS_setSpiBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_SPI_CRG) =  (HWREG(base + BISS_SPI_CRG) & \
			(~ENC_APB_CFG_BISS_SPI_BIT_NUM_M)) | (bitnum<<ENC_APB_CFG_BISS_SPI_BIT_NUM_S);
}

__STATIC_INLINE void ENC_BISS_setTDelay(uint32_t base,uint32_t tdelay)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_TIMING0) =  (HWREG(base + BISS_TIMING0) & \
			(~ENC_APB_CFG_BISS_TDLY_CFG_M)) | (tdelay<<ENC_APB_CFG_BISS_TDLY_CFG_S);
}

__STATIC_INLINE void ENC_BISS_setTCall(uint32_t base,uint32_t tcal)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_TIMING0) =  (HWREG(base + BISS_TIMING0) & \
			(~ENC_APB_CFG_BISS_TCAL_CFG_M)) | (tcal<<ENC_APB_CFG_BISS_TCAL_CFG_S);
}

__STATIC_INLINE void ENC_BISS_setTr(uint32_t base,uint32_t tr)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + BISS_TIMING1) =  (HWREG(base + BISS_TIMING1) & \
			(~ENC_APB_CFG_BISS_TR_CFG_M)) | (tr<<ENC_APB_CFG_BISS_TR_CFG_S);
}


__STATIC_INLINE uint16_t ENC_BISS_getTimeofWaitRpt(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	return ((HWREG(base + BISS_STATUS) & \
			ENC_APB_CFG_BISS_WAIT_TIME_RPT_M)>>ENC_APB_CFG_BISS_WAIT_TIME_RPT_S);
}

__STATIC_INLINE uint16_t ENC_BISS_getBissState(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	return (HWREG(base + BISS_STATUS) & ENC_APB_CFG_BISS_STATE_M);
}

//***********************************     BISS_END     ***********************************//


//***********************************     ENDAT_START     ***********************************//
__STATIC_INLINE void ENC_ENDAT_Enable(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_CTRL0) |=	ENC_APB_CFG_ENDAT_CTRL0_ENDAT_EN_M;
}

__STATIC_INLINE void ENC_ENDAT_Disable(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_CTRL0) &=	(~ENC_APB_CFG_ENDAT_CTRL0_ENDAT_EN_M);
}

__STATIC_INLINE void ENC_ENDAT_selRcvMode(uint32_t base,RCV_MODE mode)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_CTRL0) =  (HWREG(base + ENDAT_CTRL0) & \
			(~ENC_APB_CFG_ENDAT_CTRL0_RXD_IN_SEL_S)) | (mode<<ENC_APB_CFG_ENDAT_CTRL0_RCV_MODE_S);
}

__STATIC_INLINE void ENC_ENDAT_selRxdIn(uint32_t base,RXD_INPUTSIGNAL signal)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_CTRL0) =  (HWREG(base + ENDAT_CTRL0) & \
			(~ENC_APB_CFG_ENDAT_CTRL0_RXD_IN_SEL_M)) | (signal << ENC_APB_CFG_ENDAT_CTRL0_RXD_IN_SEL_S);
}

__STATIC_INLINE void ENC_ENDAT_setBaudrate(uint32_t base,uint32_t baudrate)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_CTRL0) =  (HWREG(base + ENDAT_CTRL0) & \
			(~ENC_APB_CFG_ENDAT_CTRL0_BAUDRATE_M)) | (baudrate<<ENC_APB_CFG_ENDAT_CTRL0_BAUDRATE_S);
}

__STATIC_INLINE void ENC_ENDAT_setTxBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_CTRL1) =  (HWREG(base + ENDAT_CTRL1) & \
			(~ENC_APB_CFG_ENDAT_CTRL1_ENDAT_TX_BIT_NUM_M)) | (bitnum<<ENC_APB_CFG_ENDAT_CTRL1_ENDAT_TX_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setRxBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_CTRL1) =  (HWREG(base + ENDAT_CTRL1) & \
			(~ENC_APB_CFG_ENDAT_CTRL1_ENDAT_RX_BIT_NUM_M)) | (bitnum<<ENC_APB_CFG_ENDAT_CTRL1_ENDAT_RX_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setAddBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_CTRL1) =  (HWREG(base + ENDAT_CTRL1) & \
			(~ENC_APB_CFG_ENDAT_CTRL1_ENDAT_RX_BIT_NUM_S)) | (bitnum<<ENC_APB_CFG_ENDAT_CTRL1_ENDAT_RX_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setSpiTxBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_SPI_CFG) =  (HWREG(base + ENDAT_SPI_CFG) & \
			(~ENC_APB_CFG_ENDAT_SPI_CFG_ENDAT_SPI_TX_BIT_NUM_M)) | (bitnum<<ENC_APB_CFG_ENDAT_SPI_CFG_ENDAT_SPI_TX_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setSpiRxBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_SPI_CFG) =  (HWREG(base + ENDAT_SPI_CFG) & \
			(~ENC_APB_CFG_ENDAT_SPI_CFG_ENDAT_SPI_RX_BIT_NUM_M)) | (bitnum<<ENC_APB_CFG_ENDAT_SPI_CFG_ENDAT_SPI_RX_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setSpiAddBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_SPI_CFG) =  (HWREG(base + ENDAT_SPI_CFG) & \
			(~ENC_APB_CFG_ENDAT_SPI_CFG_ENDAT_SPI_ADD_BIT_NUM_M)) | (bitnum<<ENC_APB_CFG_ENDAT_SPI_CFG_ENDAT_SPI_ADD_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setTDelay(uint32_t base,uint32_t tdelay)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_TIMING0) =  (HWREG(base + ENDAT_TIMING0) & \
			(~ENC_APB_CFG_ENDAT_TIMING0_TDLY_CFG_M)) | (tdelay<<ENC_APB_CFG_ENDAT_TIMING0_TDLY_CFG_S);
}

__STATIC_INLINE void ENC_ENDAT_setTst(uint32_t base,uint32_t tst)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_TIMING0) =  (HWREG(base + ENDAT_TIMING0) & \
			(~ENC_APB_CFG_ENDAT_TIMING0_ST_CFG_M)) | (tst<<ENC_APB_CFG_ENDAT_TIMING0_ST_CFG_S);
}

__STATIC_INLINE void ENC_ENDAT_setTr(uint32_t base,uint32_t tr)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_TIMING1) =  (HWREG(base + ENDAT_TIMING1) & \
			(~ENC_APB_CFG_ENDAT_TIMING1_TR_CFG_M)) | (tr<<ENC_APB_CFG_ENDAT_TIMING1_TR_CFG_S);
}

__STATIC_INLINE void ENC_ENDAT_setTAdd(uint32_t base,uint32_t tadd)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENDAT_TIMING1) =  (HWREG(base + ENDAT_TIMING1) & \
			(~ENC_APB_CFG_ENDAT_TIMING1_TADD_CFG_M)) | (tadd<<ENC_APB_CFG_ENDAT_TIMING1_TADD_CFG_S);
}

__STATIC_INLINE uint16_t ENC_ENDAT_getTimeofWaitRpt(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	return ((HWREG(base + ENDAT_STATUS) & \
			ENC_APB_CFG_ENDAT_WAIT_TIME_RPT_M)>>ENC_APB_CFG_ENDAT_WAIT_TIME_RPT_S);
}

__STATIC_INLINE uint16_t ENC_ENDAT_getBissState(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	return (HWREG(base + ENDAT_STATUS) & ENC_APB_CFG_ENDAT_STATE_M);
}

//***********************************     ENDAT_END     ***********************************//

__STATIC_INLINE void ENC_enableInterrupt(uint32_t base,uint32_t intFlags)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_INTR_MASK) &=  (~intFlags);
}

__STATIC_INLINE void ENC_disableInterrupt(uint32_t base,uint32_t intFlags)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_INTR_MASK) |=  intFlags;
}

__STATIC_INLINE uint16_t ENC_getIntrState(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	return (HWREG(base + ENC_INTR));
}

__STATIC_INLINE void ENC_clearIntrState(uint32_t base,uint32_t intFlags)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_INTR_RAW) |= intFlags;
}
#endif //0x30

__STATIC_INLINE uint16_t __byte(int* pdata, uint16_t index)
{
	return (((uint8_t*)pdata)[index]);
}

__STATIC_INLINE uint16_t __flip16(uint16_t data)
{
	uint32_t i;
	uint16_t tmp=0;
	for(i=0;i<16;i++)
	{
		tmp=((data>>i)&0x01)|tmp;
	    if(i<15)
	    	tmp=tmp<<1;
	}
	return (tmp);
}

__STATIC_INLINE uint8_t __flip8(uint8_t data)
{
	uint32_t i;
	uint8_t tmp=0;
	for(i=0;i<8;i++)
	{
		tmp=((data>>i)&0x01)|tmp;
	    if(i<7)
	    	tmp=tmp<<1;
	}
	return (tmp);
}
#if IS_GS32F00xx(0x30)
extern uint16_t PM_tformat_setupCommand(uint16_t dataID, uint16_t eepromAddr, uint16_t eepromData, uint16_t crc);
extern uint16_t PM_tformat_receiveData(uint16_t dataID);
extern void PM_tformat_setupPeriph(void);
extern void PM_tformat_setFreq(uint32_t Freq_us);
extern void PM_tformat_startOperation(void); 
extern void PM_tformat_generateCRCTable(uint16_t nBits, uint16_t polynomial, uint16_t *pTable);
extern uint16_t PM_tformat_getCRC(uint16_t input_crc_accum, uint16_t nBitsData,  uint16_t nBitsPoly, uint16_t * msg, uint16_t *crc_table, uint16_t rxLen);
extern void PM_tformat_setupCommandReadoutOrReset(uint16_t commandID0_1_2_3_7_8_C,uint16_t tformatRXClocks,uint16_t tformatRXFields,uint16_t tformatTXClocks,uint16_t tformatFIFOLevel);
extern void PM_tformat_setupCommandReadEEPROM(uint16_t eepromAddr);
extern void PM_tformat_setupCommandWriteEEPROM(uint16_t eepromAddr,uint16_t eepromData);
extern void PM_tformat_receiveDataID0_1_7_8_C(void);
extern void PM_tformat_receiveDataID2(void);
extern void PM_tformat_receiveDataID3(void);
extern uint16_t PM_tformat_receiveDataIDD(void);
extern uint16_t PM_tformat_receiveDataID6(void);

#endif
#endif //0

#ifdef __cplusplus
}
#endif

#endif /* DEVICE_DRIVERLIB_ENCODER_H_ */

