/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    fft.h
*   @brief   
*
*/

#ifndef DEVICE_DRIVERLIB_FFT_H_
#define DEVICE_DRIVERLIB_FFT_H_

#if GS32F00xx == 0x1200 || GS32F00xx == 0x0012

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "inc/hw_fft.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "debug.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

typedef enum {
  FFT_TYPE_COMPLEX = 0,
  FFT_TYPE_REAL = 1,
} FFT_TYPE;

/**
 * @brief Values that can be passed to FFT_SET_TYPE() as the type parameter.
 */
typedef enum {
  COMPLEX_FFT_LEN_256 = 0,
  COMPLEX_FFT_LEN_512 = 1,
  COMPLEX_FFT_LEN_1024 = 2,
  REAL_FFT_LEN_512 = 0,
  REAL_FFT_LEN_1024 = 1,
  REAL_FFT_LEN_2048 = 2,
} FFT_LEN;

typedef enum {
  FFT_DMA_TRIG_CHAN1 = 0,
  FFT_DMA_TRIG_CHAN2 = 1,
  FFT_DMA_TRIG_CHAN3 = 2,
  FFT_DMA_TRIG_CHAN4 = 3,
  FFT_DMA_TRIG_CHAN5 = 4,
  FFT_DMA_TRIG_CHAN6 = 5,
  FFT_DMA_TRIG_CHAN7 = 6,
  FFT_DMA_TRIG_CHAN8 = 7,
} FFT_DMA_TRIG_CHAN;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

/**
 * @brief Checks FFT base address.
 * @details
 * This function determines if a FFT base address is valid.
 * @param [in] ase specifies the FFT base address.
 * @return true the base address is valid
 * @return false otherwise.
 *
 */
#ifdef DEBUG
static inline boolean FFT_isBaseValid(uint32_t base)
{
    return((base == FFT_BASE));
}
#endif

/**
 * @brief   Enable FFT module to exchange the final result in the higher and the lower half of the DATA RAM.
 *
 * @param   base  FFT base address
 *
 */
static inline void FFT_EXCHANGE_RESULT_ENABLE(uint32_t base)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_CFG) |= FFT_CFG_EXCHANGE_EN_M;
}

/**
 * @brief   Disable FFT module to exchange the final result in the higher and the lower half of the DATA RAM.
 *
 * @param   base    FFT base address
 *
 */
static inline void FFT_EXCHANGE_RESULT_DISABLE(uint32_t base)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_CFG) &= ~FFT_CFG_EXCHANGE_EN_M;
}

/**
 * @brief   set FFT operation mode
 *
 * @param   base  FFT base address
 * @param   type  FFT operation mode
 *
 */
static inline void FFT_SET_TYPE(uint32_t base, FFT_TYPE type)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_CFG) = (HWREG(base + FFT_O_FFT_CFG) & ~(FFT_CFG_FFT_TYPE_M)) | (type << FFT_CFG_FFT_TYPE_S);
}

/**
 * @brief   set FFT calculation length
 *
 * @param   base    FFT base address
 * @param   len     FFT calculation data points
 *
 */
static inline void FFT_SET_LEN(uint32_t base, FFT_LEN len)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_CFG) = (HWREG(base + FFT_O_FFT_CFG) & ~(FFT_CFG_FFT_LEN_M)) | (len << FFT_CFG_FFT_LEN_S);
}

/**
 * @brief   FFT windowing function software start
 *
 * @param   base    FFT base address
 *
 */
static inline void FFT_WIN_SOFT_START(uint32_t base)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_START) = FFT_START_WIN_SOFT_START;        //W1SR0
}

/**
 * @brief   FFT calculation software start
 *
 * @param   base    FFT base address
 *
 */
static inline void FFT_SOFT_START(uint32_t base)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_START) = FFT_START_FFT_SOFT_START;       //W1SR0
}

/**
 * @brief   enable DSP DMA channel interrupt triggering FFT start
 *
 * @param   base    FFT base address
 * @param   chan    DSP DMA channel
 *
 */
static inline void FFT_DSPDMA_TRIG_ENABLE(uint32_t base, FFT_DMA_TRIG_CHAN chan)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_DSPDMATRIG_CFG) |= 1 << chan;
}

/**
 * @brief   disable DSP DMA channel interrupt triggering FFT start
 *
 * @param   base    FFT base address
 * @param   chan    DSP DMA channel
 */
static inline void FFT_DSPDMA_TRIG_DISABLE(uint32_t base, FFT_DMA_TRIG_CHAN chan)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_DSPDMATRIG_CFG) &= ~(1 << chan);
}

/**
 * @brief   enable FFT windowing function
 *
 * @param   base    FFT base address
 *
 */
static inline void FFT_WIN_ENABLE(uint32_t base)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_AUTOSTART_EN) &= ~(FFT_WIN_BYPASS);
}

/**
 * @brief   disable FFT windowing function
 *
 * @param   base    FFT base address
 *
 */
static inline void FFT_WIN_DISABLE(uint32_t base)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_AUTOSTART_EN) |= (FFT_WIN_BYPASS);
}

/**
 * @brief   enable window FFT calculation triggered by hardware (auto start)
 *
 * @param   base    FFT base address
 *
 */
static inline void FFT_WIN_AUTOSTART_ENABLE(uint32_t base)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_AUTOSTART_EN) |= (FFT_WIN_AUTOSTART_EN);
}

/**
 * @brief   disable window FFT calculation triggered by hardware (auto start)
 *
 * @param   base    FFT base address
 *
 */
static inline void FFT_WIN_AUTOSTART_DISABLE(uint32_t base)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_AUTOSTART_EN) &= ~(FFT_WIN_AUTOSTART_EN);
}

/**
 * @brief   enable FFT calculation triggered by hardware (auto start)
 *
 * @param   base    FFT base address
 */
static inline void FFT_AUTOSTART_ENABLE(uint32_t base)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_AUTOSTART_EN) |= (FFT_FFT_AUTOSTART_EN);
}

/**
 * @brief   disable FFT calculation triggered by hardware (auto start)
 *
 * @param   base    FFT base address
 */
static inline void FFT_AUTOSTART_DISABLE(uint32_t base)
{
    ASSERT(FFT_isBaseValid(base));

    HWREG(base + FFT_O_FFT_AUTOSTART_EN) &= ~(FFT_FFT_AUTOSTART_EN);
}

/**
 * @brief   check if window FFT is busy
 *
 * @param   base    FFT base address
 *
 * @return  true    window FFT is busy
 */
static inline boolean FFT_IS_WIN_BUSY(uint32_t base)
{
    return (HWREG(base + FFT_O_FFT_STATUS) & FFT_STATUS_WIN_BUSY) != 0;
}

/**
 * @brief   check if FFT is busy
 *
 * @param   base    FFT base address
 *
 * @retuen  true   FFT is busy
 */
static inline boolean FFT_IS_BUSY(uint32_t base)
{
    return (HWREG(base + FFT_O_FFT_STATUS) & FFT_STATUS_FFT_BUSY) != 0;
}

/**
 * @brief   enable fft interrupt
 *
 * @param   base    FFT base address
 * @param   mask   FFT interrupt mask
 *          value  FFT_OVF_INT_MASK, FFT_DONE_INT_MASK
 *
 */
static inline void FFT_INTERRUPT_ENABLE(uint32_t base, uint32_t mask)
{
    HWREG(base + FFT_O_FFT_INT_MASK) &= ~mask;
}

/**
 * @brief   disable fft interrupt
 *
 * @param   base    FFT base address
 * @param   mask   FFT interrupt mask
 *          value  FFT_OVF_INT_MASK, FFT_DONE_INT_MASK
 *
 */
static inline void FFT_INTERRUPT_DISABLE(uint32_t base, uint32_t mask)
{
    HWREG(base + FFT_O_FFT_INT_MASK) |= mask;
}

/**
 * @brief   get fft interrupt flag
 *
 * @param   base    FFT base address
 *
 * @return  FFT_INT_FLG_OVF_INT, FFT_INT_FLG_DONE_INT, FFT_INT_FLG_INT
 */
static inline uint32_t FFT_GET_INTERRUPT_FLAG(uint32_t base)
{
    return HWREG(base + FFT_O_FFT_INT_FLG);
}

/**
 * @brief   clear fft interrupt
 *
 * @param   base    FFT base address
 * @param   flag    FFT interrupt flag
 *          values FFT_INT_CLR_OVF_INT_CLR, FFT_INT_CLR_DONE_INT_CLR
 *
 */
static inline void FFT_CLEAR_INTERRUPT_FLAG(uint32_t base, uint32_t flag)
{
    HWREG(base + FFT_O_FFT_INT_RAW) = flag;
}

/**
 * @brief   fource trigger fft interrupt by software
 *
 * @param   base    FFT base address
 * @param   flag   FFT interrupt flag
 *          value  FFT_INT_FRC_OVF_INT_FRC, FFT_INT_FRC_DONE_INT_FRC
 *
 */
static inline void FFT_FOURCE_TRIG_INTERRUPT(uint32_t base, uint32_t flag)
{
    HWREG(base + FFT_O_FFT_INT_FRC) = flag;
}

/**
 * @brief   enable window FFT interrupt
 *
 * @param   base    FFT base address
 * @param   parameter2   description of parameter2
 *
 */
static inline void FFT_WIN_INTERRUPT_ENABLE(uint32_t base)
{
    HWREG(base + FFT_O_WIN_INT_MASK) &= ~(FFT_WIN_DONE_INT_MASK);
}

/**
 * @brief   disable window FFT interrupt
 *
 * @param   base   FFT base address
 *
 */
static inline void FFT_WIN_INTERRUPT_DISABLE(uint32_t base)
{
    HWREG(base + FFT_O_WIN_INT_MASK) |= FFT_WIN_DONE_INT_MASK;
}

/**
 * @brief   get window FFT interrupt flag
 *
 * @param   base    FFT base address
 *
 * @return  window FFT flag
 */
static inline uint32_t FFT_GET_WIN_INTERRUPT(uint32_t base)
{
    return HWREG(base + FFT_O_WIN_INT_FLG);
}

/**
 * @brief   clear window FFT interrupt flag
 *
 * @param   base    FFT base address
 *
 */
static inline void FFT_CLEAR_WIN_INTERRUPT(uint32_t base)
{
    HWREG(base + FFT_O_WIN_INT_RAW) = FFT_WIN_DONE_INT_CLR;
}

/**
 * @brief   fource trigger fft window interrupt by software
 *
 * @param   base    FFT base address
 *
 */
static inline void FFT_FOURCE_TRIG_WIN_INTERRUPT(uint32_t base)
{
    HWREG(base + FFT_O_WIN_INT_FRC) = FFT_WIN_DONE_INT_FRC;
}
#ifdef __cplusplus
}
#endif

#endif  /* GS32F00xx == 0x1200 || GS32F00xx == 0x0012 */

#endif /* DEVICE_DRIVERLIB_FFT_H_ */
