/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __STL_ANALOG_IO_H__
#define __STL_ANALOG_IO_H__

#include <stdint.h>

#include "stl_err.h"
#include "driverlib.h"

/**
 * @brief Analog IO input test(ADC).
 * 
 * @param base The base address of ADC instance.
 * @param result_base The base address of ADC result instance.
 * @param soc_num Start of convert sequence number.
 * @param chl ADC channel number.
 * @param clk_div ADC clock divider.
 * @param ref_voltage_mode Select ADC reference voltage(external mode or internal mode).
 * @param level ADC reference voltage level.
 * @param expectation_val The expected value of ADC result.
 * @param reference_voltage The reference voltage value of ADC.
 * If select external reference voltage, this parameter is not used.
 * @return stl_err_t If the ADC test is successful, return STL_SUCCESS.
 * otherwise return non-zero values.
 */
stl_err_t stl_analog_io_input(uint32_t base, uint32_t result_base,
							ADC_SOCNumber soc_num, ADC_Channel chl,
							ADC_ClkPrescale clk_div, ADC_ReferenceMode ref_voltage_mode,
							ADC_ReferenceVoltage level, float expectation_val,
							float reference_voltage);

/**
 * @brief Analog IO output test(DAC). 
 * Use ADC and DAC internal loopback mode to test DAC.
 * The maximum output voltage value of the DAC is half of the DAC reference voltage.
 * 
 * @param base The base address of DAC instance.
 * @param ref_voltage_mode DAC reference voltage mode.
 * @param loadmode DAC load mode.
 * @param gainmode DAC Gain mode.
 * @param expectation_val DAC output expectated value.
 * @param reference_voltage DAC reference voltage value.
 * @param adc_soc_num ADC SOC number.
 * @param adc_ref_voltage_mode ADC reference voltage mode.
 * @param adc_reference_voltage ADC reference voltage value.
 * @return stl_err_t If the DAC test is successful, return STL_SUCCESS.
 * otherwise return non-zero values.
 */
stl_err_t stl_analog_io_output(uint32_t base, DAC_ReferenceVoltage ref_voltage_mode,
							DAC_LoadMode loadmode, DAC_GainMode gainmode,
							float expectation_val, float reference_voltage,
							ADC_SOCNumber adc_soc_num, ADC_ReferenceMode adc_ref_voltage_mode,
							float adc_reference_voltage);

#endif
