/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __STL_DIGITAL_IO_H__
#define __STL_DIGITAL_IO_H__

#include <stdint.h>

#include "stl_err.h"

/**
 * @brief Analog IO as digital IO function to capture IO level.
 * 
 * @param io_num GPIO number.
 * @param expectation_value High level or low level.
 * @param delay_cnt convert time.
 * @return stl_err_t If the input IO level is not as expected, return non-zero values.
 * else it is success.
 */
stl_err_t stl_digital_aio_input(uint16_t io_num,
							uint16_t expectation_value,
							uint32_t delay_cnt);

/**
 * @brief Digital IO to capture IO level.
 * 
 * @param io_num GPIO number.
 * @param expectation_value High level or low level.
 * @param delay_cnt convert time.
 * @return stl_err_t If the input IO level is not as expected, return non-zero values.
 * else it is success.
 */
stl_err_t stl_digital_io_input(uint16_t io_num,
							uint16_t expectation_value,
							uint32_t delay_cnt);

/**
 * @brief Analog IO as digital IO function to output IO level.
 * 
 * @param io_num GPIO number.
 * @return stl_err_t If return non-zero values, it is failed. else successed.
 */
stl_err_t stl_digital_aio_output(uint16_t io_num);

/**
 * @brief Digital IO to output IO level.
 * 
 * @param io_num GPIO number.
 * @return stl_err_t If return non-zero values, it is failed. else successed.
 */
stl_err_t stl_digital_io_output(uint16_t io_num);

#endif
