/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "stl_clock.h"
#include "stl_err.h"
#include "interrupt.h"
#include "nmsis_core.h"

volatile uint32_t stl_coreTicks = 0;

volatile uint32_t stl_cpu_frequency;

void eclic_mtip_handler(void)
{
	uint64_t loadticks = SysTimer_GetLoadValue();
	SysTimer_SetCompareValue(stl_cpu_frequency/1000 + loadticks);

	stl_coreTicks += 1;
}

void stl_delay_ms(uint32_t cpu_freq, uint32_t ms)
{
	uint64_t cpu_cnt_ms;
	uint64_t read_ticks, ticks_current;

	read_ticks = __get_rv_time();

	cpu_cnt_ms = cpu_freq / 1000;

	cpu_cnt_ms = cpu_cnt_ms * ms;

	while (1) {
		ticks_current = __get_rv_time();

		if (ticks_current - read_ticks >= cpu_cnt_ms)
			break;
	}
}

stl_err_t stl_clock_test(uint32_t cpu_freq)
{
	uint32_t cmp = 0;
	uint32_t ms = 1;

	stl_cpu_frequency = cpu_freq;

	SysTimer->MTIMECTL = SysTimer_MTIMECTL_CLKSRC_Msk;
	Interrupt_register(SysTimer_IRQn, &eclic_mtip_handler);
	SysTick_Config(stl_cpu_frequency / 1000);

	/* Wait for 1 second */
	stl_delay_ms(cpu_freq, 1);

	ECLIC_DisableIRQ(SysTimer_IRQn);

	if (stl_coreTicks > ms)
		cmp = stl_coreTicks - ms;
	else if (stl_coreTicks < ms)
		cmp = ms - stl_coreTicks;

	if (cmp > 1)
		return STL_ERR_CHECK_FAILED;

	return STL_SUCCESS;
}
